rng = Random.default_rng()
Random.seed!(99)

C = 0.01


#Initial Conditions
I = [1.0, 0.0]   #Psi(0)=1, Psi'(0)=1
etaspan = (0.05, 5.325)

#radius range
datasize= 100
etasteps = range(etaspan[1], etaspan[2]; length = datasize)


function whitedwarf(du, u, p, r)
    psi = u[1]
    dpsi = u[2]
    du[1] = dpsi
    du[2] = (-((psi^2-C))^(3/2) - 2/r * dpsi)
end


#Defining the Ordinary differential equation as an ODEProblem with the DifferentialEquations.jl
prob = ODEProblem(whitedwarf, I, etaspan)
#Solving the ODEProblem with the Tsit5() algorithm
sol = solve(prob,saveat=etasteps)

dudt2 = Lux.Chain(Lux.Dense(2, 80, tanh),Lux.Dense(80, 80, tanh), Lux.Dense(80, 2))
#Setting up the NN parameters randomly using the rng instance
p, st = Lux.setup(rng, dudt2)

etasteps =  etasteps[1:end-20]
etaspan = (etasteps[1], etasteps[end])
I=[1.0,0.0]
prob_neuralode = NeuralODE(dudt2, etaspan, Tsit5(); saveat = etasteps)

function predict_neuralode(p)
    Array(prob_neuralode(I, p, st)[1])
end
#Training data
true_data= Array(sol[:,1:end-20])
### Define loss function as the difference between actual ground truth data and Neural ODE prediction

function loss_neuralode(p)
    pred = predict_neuralode(p)
    loss = sum(abs2, true_data .- pred)
    return loss, pred
end

p = (layer_1 = (weight = Float32[-0.5371412 0.46584868; -0.82070273 -0.6242048; -0.8492154 -0.048775695; -0.2955019 -1.9671385; -0.6853695 0.6758744; -0.14646325 -0.9450923; 0.42263308 -0.49178642; -0.28922072 -0.5128624; -1.0927844 0.23534705; 0.31981084 2.1595411; -1.1345546 0.3576599; 0.7251795 -0.69696707; 0.59820426 -0.19346781; 0.51383114 -1.3474408; 0.44247425 -0.07944205; 0.87030655 -0.36628717; 0.31992912 0.18661815; 0.8337897 -0.9469921; 0.5032956 -0.1246921; 0.39966607 0.580603; -0.9124741 0.5084461; 1.0027435 -0.4996795; -1.2568315 0.12824304; 0.56036526 0.059562825; 0.744622 -0.16656153; -0.18797155 0.772838; -0.5108172 -0.062057957; 0.37962064 0.40832648; 0.41341585 -0.50538146; 1.1067274 1.0351236; 0.7645613 0.46762604; -0.6122835 -0.119906366; 0.7935727 -0.09355478; -0.98938185 0.8658866; -0.4632263 -0.78548783; 0.6659643 -0.5602622; -0.23689386 -0.17221044; 0.6127161 1.7400867; 1.0617125 -0.10380499; 0.5464079 -2.2176268; -1.0412625 0.18156184; -0.6538864 0.015470294; -0.3574077 -0.75934935; -0.59984195 -0.045256507; -0.35653052 1.3919085; 0.76348406 -0.54798573; -0.95982915 0.83337045; -0.46313652 -0.4238968; 0.33315074 -0.57240874; 1.0749645 -0.70094013; -1.1432108 0.3460136; 0.80214024 -0.23956898; -0.4105014 -0.74160635; 0.5675491 0.16424613; -0.6941691 0.08495913; 0.18317099 -0.33373335; 0.7409308 -0.1584677; -0.7639061 0.12907082; -0.81181 0.08840492; -0.24526812 1.8860724; 0.31900364 0.31465587; -0.5033191 0.16595498; 0.6695171 -0.7191454; -0.25567088 0.7246624; -0.4064015 -0.5195756; 0.7925136 0.19652897; 1.0679233 -0.044480223; 1.3112164 0.13858488; -1.0809189 0.47981414; 0.75146526 0.5910382; 1.1480689 -0.17357235; 0.8087713 -0.8091088; -0.73956954 0.13868394; -0.4272903 -0.52873987; -0.30868155 -0.7176204; 0.35798982 1.6778541; 0.90672815 -0.17841592; 0.13996646 0.95413035; -0.49110615 0.1059229; -0.25112292 -0.7443114], bias = Float32[-1.1443869; 0.3323404; 0.718776; -0.9735976; -1.0149833; 0.8697972; 1.0662824; -0.3394195; -0.50796974; 1.1183572; -0.57497627; 0.9388404; -0.97847044; 0.2926941; -0.37582535; 0.52418303; 1.1322361; 0.64543164; 0.817839; 0.38764846; -0.15362024; 0.23094752; -0.89991236; 1.295672; 0.5772173; 0.012883152; -1.0881554; 1.2201103; -0.66515046; 0.55436695; 1.066916; -0.835968; 1.299622; -0.6823734; -0.61919403; 1.0996597; -1.1183678; 0.983654; 0.86418134; 0.57119477; -0.69596654; -0.9682082; 0.65283096; -0.775301; 0.27452406; 1.090619; -0.72167844; -0.4415398; 1.0335526; 0.38462332; -0.57219964; 1.2856151; -0.71786207; 0.7951713; 0.8358842; -0.95448196; 0.5296833; -0.95597285; -0.6267931; -0.5151182; 0.8562954; -1.4255184; 0.80972415; 0.101047926; -0.7309538; -0.26850447; 0.62229127; 1.0461166; -0.3076642; -0.4177126; 0.55865824; 0.7310776; -0.57385284; 0.9604619; -0.71282464; 0.3709177; 0.53820676; 0.3587505; -1.3270752; -0.88501185]), layer_2 = (weight = Float32[-0.8950642 0.8345793 0.7021911 -0.14260204 -0.9044946 0.88751954 0.7491279 -0.7328375 0.83532006 0.57560676 -0.26075304 0.7653368 -0.80209506 0.42054972 0.49189726 -0.497827 0.17193379 0.52528834 0.797071 0.73105544 -0.15479091 0.06872221 -0.3122765 0.60635746 -0.828258 0.775683 -0.34598917 0.5772039 -0.26716334 -0.80106884 0.9038309 0.41561973 0.61070836 -0.13556442 0.13899037 0.9640773 -0.33365685 0.4788852 0.325053 -0.077473626 -0.45809367 0.15906046 0.64119345 0.35500342 -0.25136137 0.71953803 -0.00061054475 -0.81598973 0.9465403 -0.23316243 0.5774731 0.79978365 -0.87114555 -0.34467107 0.8487575 -0.24351776 -0.5814821 -0.0875792 0.3633184 -0.42605838 -0.28576872 -0.8681717 0.27497813 0.712339 0.12470286 -0.9057785 -0.72552085 -0.65050113 0.8277516 -0.6224311 -0.7582386 -0.20140629 0.9302355 0.69102985 -0.31185445 0.020622669 0.34847227 0.8130812 -0.7664091 -0.69701374; -0.49268737 0.4075955 0.50638497 -0.7311153 -0.7572168 0.6148531 0.51631385 -0.10183383 0.33576414 0.72769105 0.5096774 0.095301084 -0.528287 -0.60724586 -0.7167681 -0.4585179 0.8187961 -0.6402092 0.5647753 0.40643865 0.7847166 -0.5005241 0.5616144 0.6006002 -0.63978964 0.5169752 -0.6513121 0.5644956 -0.4617474 -0.5259821 0.4466396 -0.6134161 0.65854836 0.58433723 -0.71989304 0.50618714 -0.5064292 0.55638516 -0.70664096 -0.834329 0.71549964 -0.441067 0.35042754 -0.53278434 0.5896491 0.3918142 0.62405384 -0.3926926 0.5601347 -0.6905346 0.34487572 0.5392323 -0.46896517 0.76164085 0.51309955 -0.5434105 -0.7036105 -0.7509977 -0.6416125 0.8092818 0.5943085 -0.7126738 -0.85797817 0.6413149 -0.8014583 -0.64742565 -0.36921114 -0.4461418 0.7141169 -0.7210959 -0.5985428 -0.57060975 0.58355325 0.46341094 -0.6031567 0.5802598 -0.87988836 0.658729 -0.76760304 -0.763571; -0.2604241 0.48114765 -0.10272227 0.71405935 -0.32145613 0.45271766 0.48414934 -0.46777952 0.34365004 -0.7193702 -0.5599509 0.39988253 0.30151725 0.49828923 0.47935554 -0.55995935 -0.9352898 0.4408364 0.3568775 0.2566033 -0.7954216 0.5888458 -0.56799525 -0.8584118 -0.5925409 0.44096503 0.71570146 -0.48916036 0.5965562 -0.39593568 0.22645204 0.58886516 -0.7700896 -0.76708573 0.5704696 0.13612701 0.7980155 -0.62835455 0.6328186 0.71824646 -0.44879547 0.89048356 0.4776864 0.65620667 -0.64374614 0.20876995 -0.77668923 -0.2946555 0.18677126 0.5348476 0.34824282 0.06700021 -0.22729242 -0.90095425 -0.18711127 0.7300223 -0.18035428 0.8953346 0.5667613 -0.7502866 -0.6125264 0.1297908 0.6916505 0.2769056 0.6674936 -0.33925635 -0.4848281 0.67590165 0.24158022 -0.49527064 -0.37761778 0.4484484 0.62663853 0.39571217 0.73764527 -0.6948442 0.5443441 0.25266388 0.08306829 0.12842944; 0.060633857 -0.012253005 -0.32877818 0.918026 -0.2828569 0.38973463 -0.119994156 -0.5471017 -0.35260966 -0.59774065 -1.0785922 0.61157125 0.63957584 0.84969 0.56346273 0.0009415484 -0.56338894 0.91580534 0.28652307 0.4304038 -0.9291529 1.0185131 -1.083199 -0.6336704 -0.3226842 -0.30169865 0.741477 -0.28266954 0.8364516 0.22713734 0.60116863 0.51483953 -0.294701 -1.1751845 0.5720587 -0.04738041 0.6950297 -0.5195933 0.8565607 0.72853893 -0.97382873 0.53149307 0.029677982 0.8568592 -0.91094506 0.49206692 -0.8232908 -0.29318586 0.20881954 0.86231 -0.21913826 -0.13649064 -0.12155216 -0.77899927 -0.42828497 0.8443198 0.70747054 0.37793267 0.21869998 -0.7475928 -0.8628972 0.515307 0.9967581 0.031777386 0.75406015 0.46032834 0.15481816 0.6894724 -0.45438328 -0.25793657 0.025714412 1.127043 -0.25839278 -0.10048205 0.6393916 -0.81401163 0.92732453 -0.17070922 0.32812223 0.29109183; 0.39232603 -0.14771852 0.2080349 -0.9948371 0.30752513 -0.4809692 -0.43245313 0.44275215 -0.18837431 0.8472639 0.8294414 -0.70028913 -0.086585864 -0.7732212 -0.711768 0.21139635 0.7675354 -0.67622566 -0.5459226 -0.4103773 0.8898018 -0.7089105 0.8246188 0.89514756 0.3768983 0.0038075433 -0.6667581 0.42736715 -0.63477933 0.21944441 -0.5435328 -0.6542617 0.72981 0.9120442 -0.8013432 -0.25857493 -0.8264324 0.704828 -0.9292866 -0.7763763 0.74302787 -0.62017727 -0.51831186 -0.7067651 0.89041847 -0.54071665 1.0590231 0.48407102 -0.3212036 -0.8695831 0.0656612 0.2638526 0.43482262 0.6939421 0.14723036 -0.59515125 -0.02934294 -0.5562697 -0.3193275 0.66558754 0.5362499 -0.07862686 -0.7722802 -0.113882765 -0.6894775 0.13536987 0.30894646 -0.6779666 0.11663881 0.21764934 0.31698045 -0.88877857 -0.12535769 -0.13845603 -0.7456121 0.88621926 -0.7290614 -0.21983771 -0.28593555 -0.069678195; 0.97492456 -0.88427985 -0.87982684 0.1334287 0.61687666 -0.5310185 -0.60383964 0.7401955 -0.7099397 -0.46697542 0.25919616 -0.49699906 0.66749847 -0.33944303 -0.37635973 0.8413419 -0.08504142 -0.28288916 -0.5436565 -0.94322306 0.48156908 -0.16362238 0.027372858 -0.2644712 0.7532983 -1.0564711 -0.078631975 -0.75570196 0.38509417 1.0236422 -0.7635986 -0.39656582 -0.37977675 0.17597534 -0.51931095 -0.6482912 -0.06464276 -0.80756515 -0.36918777 -0.014112451 0.31245413 -0.23225135 -0.85631514 -0.40468118 0.1058514 -0.89393187 -0.08634146 0.65586424 -0.69017065 0.24965589 -1.0332521 -0.69887847 0.7417074 0.4806301 -0.7354886 0.35040784 0.6880683 -0.13652794 -0.671108 0.45927456 0.14350843 0.5864332 -0.35776997 -0.8292866 -0.33053085 0.90307105 1.0400988 0.8618959 -0.9211036 0.67259747 0.70104825 -0.022282796 -0.76137733 -0.9120501 0.17833124 -0.042993728 -0.29795966 -0.94241303 0.56201744 0.7261567; 0.8376508 -0.7840095 -0.7961383 -0.57322174 0.6244081 -0.82424676 -0.7688182 0.43722427 -0.48511228 0.15736632 0.6086992 -0.90551645 0.7237915 -0.47129342 -0.4530484 0.36731163 0.07643957 -0.5871292 -0.7094905 -0.60666996 0.5445046 -0.2967412 0.5335599 0.1309321 0.6232356 -0.47361112 -0.3792975 -0.7046553 -0.24175015 0.50933933 -0.8570821 -0.35232767 -0.40844655 0.4550497 -0.47852737 -0.85330504 -0.056118898 -0.5860377 -0.603987 -0.14215893 0.7207702 -0.30363968 -0.60914224 -0.18963471 0.47561434 -0.87249535 0.42552894 0.56469375 -0.6729642 -0.43550047 -0.62742585 -0.9900855 0.88614905 0.20267364 -0.86665475 -0.44044232 0.63142157 -0.0716438 -0.37748304 0.392772 0.21428172 1.1184477 -0.6368183 -0.67588174 -0.48877555 0.6003315 0.48181897 0.6031272 -0.48993897 0.49024305 0.38507017 -0.3852783 -0.5505601 -0.89767337 -0.14298525 0.53367096 -0.62522215 -0.69552493 1.0348454 0.78195536; -0.9750355 0.38409165 0.6673555 0.5093567 -0.96092016 0.6353675 0.8011861 -0.30138692 0.19706506 -0.46240047 -0.8259434 0.6358226 -0.8178803 0.74818134 0.35661975 -0.19115804 -0.32154882 0.92085046 0.8178573 0.77808326 -0.39393708 0.591323 -0.59015983 0.10980956 -0.4934067 0.5097973 0.18935128 1.0795714 0.22863601 -0.42115733 0.74097437 0.28932834 0.316641 -0.91827697 0.17776726 1.0959624 0.3310931 0.36305615 0.71468323 0.1876853 -0.5145589 -0.012506012 0.7110405 0.34916312 -0.7230425 1.1517658 -0.8766212 -0.786526 1.039227 0.3643126 0.21753338 1.2666626 -0.7348949 0.032886017 0.77815956 0.29109094 -0.1748139 0.017799618 0.09600756 -0.21432602 -0.35652882 -1.099345 0.6380932 0.36399764 0.24423319 -0.6501681 -0.5061874 -0.3402194 0.36954013 -0.5297685 -0.40675402 0.942455 0.20748995 0.75331944 0.26260445 -0.5897237 0.56458145 0.5890456 -0.98884493 -0.97781765; -0.9008064 0.603624 0.54012364 -0.40240237 -1.0035332 0.9502482 0.8333437 -0.6763985 0.36291537 0.57090783 0.38088098 1.0329009 -0.6427222 -0.3610989 -0.013799137 -0.42829052 0.7715298 -0.12922193 1.0248244 0.64967704 0.14846517 -0.34588945 0.35467923 0.7745346 -0.6842945 0.59115887 -0.58464885 0.93769974 -0.43449214 -0.36663595 0.863483 -0.587542 0.5999294 0.16738388 -0.43373126 0.75122106 -0.61583304 0.6721492 -0.013665691 -0.5564618 0.11662545 -0.53706574 0.7181975 -0.6105535 0.2490916 0.7933865 0.51220554 -0.8335969 0.8781028 -0.5166235 0.5200179 0.9214418 -0.76740646 0.3331795 0.52003324 -0.75955075 -0.5657448 -0.76767635 0.46088782 0.41929427 0.5317033 -0.7818184 -0.001296826 0.6106804 -0.64519256 -0.42587474 -0.30027932 -0.22276554 0.45185181 -0.7300832 -0.46200296 -0.15778226 0.4026617 0.5568971 -0.8712783 0.60846597 0.48084015 0.50466985 -0.8446207 -0.7484199; -0.0024498084 0.011161571 0.66675603 -0.85787266 0.26565507 -0.21619105 -0.16961621 0.28149092 0.34565443 0.67424434 1.0165632 -0.64786714 -0.45471343 -0.91455996 -0.85769993 0.21730177 0.51596695 -0.9308434 -0.16896237 -0.34997907 0.6625359 -1.0472826 0.9278283 0.44734505 0.27109265 0.052479204 -0.47953838 0.4627751 -0.850847 -0.3444814 -0.35998133 -0.78195786 0.35803893 0.8525506 -0.49343956 -0.07305048 -0.6478551 0.54024863 -0.8716063 -0.8604504 1.0464501 -0.71889615 -0.01983426 -0.75633 0.86686325 -0.17906964 0.842943 0.42949483 -0.11949981 -0.9852408 0.16901076 0.5919419 0.049804207 0.61520725 0.6478658 -0.8148533 -0.53754216 -0.78656214 -0.14451262 0.50967073 0.8280437 -0.4705386 -1.0338928 0.11327052 -0.63032097 -0.26870924 -0.28304613 -0.6928186 0.49591127 0.16767724 -0.11888676 -1.1326504 0.17674904 -0.01956763 -0.63902766 0.74031264 -0.5540612 -0.114910714 -0.46638313 -0.45839778; -0.40112442 0.38216496 -0.2040987 0.99743456 -0.5754236 0.48054093 0.28051785 -0.48255676 0.37938768 -1.0336269 -0.77103937 0.65512055 0.5042625 0.75000316 0.6042144 -0.19731133 -0.6740324 0.8640281 0.26811957 0.36999607 -0.6904971 0.8996669 -0.7598013 -0.6459554 -0.37067038 0.04845717 0.64057255 -0.6046522 0.7592895 -0.44395396 0.5553515 0.74799246 -0.45222726 -0.7212737 0.6505487 0.4264437 0.73660696 -0.59310395 0.59780246 0.94953847 -0.7331979 0.5890199 0.5735636 0.72048795 -0.9651517 0.5591671 -1.0348412 -0.48981166 0.4023978 0.8439194 0.0849166 -0.023665363 -0.2503729 -0.83198106 -0.048273653 0.87509954 0.11262221 0.7060928 0.47909632 -0.8300507 -0.836344 0.22443895 0.8076032 0.40670818 0.7862116 -0.31553352 -0.23072332 0.6599033 -0.053519964 -0.53537774 -0.33166912 1.038277 0.3688553 0.22233318 0.7016843 -0.9346331 0.58000475 0.21006559 0.15910242 0.32495868; 0.21655384 -0.5425038 0.012602863 -0.9231975 0.15504971 -0.5595991 -0.27083975 0.30625102 -0.48585516 0.7769592 0.50005615 -0.25074488 -0.27097884 -0.53078 -0.82249665 0.511213 1.0677742 -0.46418038 -0.22025384 -0.2817152 0.5661897 -0.6817698 0.6173906 0.91372144 0.5467977 -0.44638374 -0.83586943 0.5049814 -0.80453265 0.55471647 -0.11548765 -0.78236467 0.80198497 0.4709109 -0.7842346 -0.06887574 -0.92504424 0.7857895 -0.6825761 -0.8304304 0.6198529 -0.940173 -0.42861837 -0.7433044 0.71913487 -0.08023716 0.6742069 0.43181622 -0.13403213 -0.6957497 -0.5663822 0.16028431 0.33790892 0.7379715 0.22486445 -0.89685804 0.0016525212 -0.8385832 -0.74264604 0.5370694 0.82318926 -0.29831517 -0.7218771 -0.39034337 -0.6870488 0.09621833 0.3457715 -0.26686588 -0.061776157 0.49760777 0.7064324 -0.76242495 -0.5297105 -0.46941632 -0.83804107 0.74530447 -0.675905 -0.14595143 -0.24846257 -0.23744059; -0.6489394 1.0479962 0.6992627 -0.2954358 -0.6864074 0.7706134 0.70052856 -0.6615696 1.0544784 0.416184 0.02745768 0.5872298 -0.8760658 0.13840856 0.30139002 -0.83662015 0.29996693 0.053237807 0.84535146 0.81266767 -0.032734856 -0.19182298 0.3106547 0.6112715 -0.81941557 0.9673013 -0.34177488 0.7381435 -0.45109844 -1.0396032 0.8236104 0.25842935 0.6698634 0.34772953 0.05383645 0.70799094 -0.44894305 0.6522521 0.063301824 -0.16759843 -0.2672572 0.15634184 0.99643505 0.09495298 0.4218196 0.9368996 0.5312414 -0.9079946 0.6105029 -0.49085236 0.89716065 0.45418847 -0.93540394 -0.46837726 0.82672125 -0.6182718 -0.65499866 -0.06817106 0.4943294 -0.24998865 0.067177676 -0.6710292 0.14767042 0.9326154 0.028344577 -0.788898 -0.75623137 -0.8817999 0.7360357 -0.6657212 -0.892806 -0.4187137 1.0652183 0.969773 -0.47862038 0.18383002 0.51807654 0.775755 -0.46066788 -0.6627626; -0.89939535 0.61202925 0.7276274 0.47506085 -0.777531 0.85836625 0.5325487 -0.65267426 0.76062363 0.25530913 -0.34261864 0.7401311 -1.0632235 0.37478256 0.5761716 -0.8686921 -0.4032387 0.5298358 0.5227531 0.62477016 -0.33243173 0.47006664 -0.54127884 -0.079256706 -0.6019007 0.6429175 0.44938248 0.69364476 0.13509643 -0.79387486 0.6944935 0.6318739 0.2170524 -0.17156492 0.48124844 0.8676876 0.36646733 0.54455477 0.23288238 0.4810046 -0.3062766 0.55003005 0.64225376 0.37186706 -0.23844394 0.47115138 -0.115354896 -0.6510147 0.7707991 0.1988062 0.9791084 0.49388757 -0.85213 -0.44605905 0.9856547 0.20650233 -0.83813435 0.51176006 0.35812616 -0.66253537 -0.43955117 -0.7763309 0.4915627 0.63270783 0.47546792 -0.77181005 -0.80592847 -1.0633922 0.936318 -0.583597 -0.6222064 0.13581751 0.81406647 0.774602 0.12944885 -0.27499664 0.6055587 0.77873206 -0.7674168 -0.9617789; 0.59361625 -0.6963512 -0.7037576 0.5830701 0.6072236 -0.61536086 -0.64019597 0.54752845 -0.4987516 -0.7812138 -0.5807596 -0.6617228 0.57683396 0.5458627 0.653978 0.6003456 -0.82536674 0.5701382 -0.61929095 -0.54321 -0.5413996 0.70633715 -0.37282142 -0.5391354 0.62460536 -0.39380485 0.64830893 -0.70476556 0.55433476 0.64117044 -0.5894156 0.6305826 -0.67289144 -0.45874146 0.67572147 -0.6799211 0.73738647 -0.5324979 0.4448106 0.48394987 -0.45714688 0.4736634 -0.57030976 0.5304948 -0.6286157 -0.6312756 -0.558015 0.47056848 -0.5718584 0.65159434 -0.59578055 -0.8101341 0.53980076 -0.68371445 -0.51454234 0.6346246 0.6214212 0.7620953 -0.5351198 -0.7887555 -0.7176313 0.7785535 0.56345284 -0.53607726 0.6819793 0.75316584 0.3846245 0.614325 -0.40260407 0.6786871 0.5895039 0.6814428 -0.6154243 -0.64946616 0.7893613 -0.55583 0.13408868 -0.79010457 0.7897394 0.68716997; 0.9648986 -0.5808339 -0.7329447 0.20502725 0.91498095 -0.6872148 -0.87112767 0.6569354 -0.5863851 -0.68585056 0.17197697 -0.75917804 0.5093225 -0.58912265 -0.22156726 0.45040846 -0.65884995 -0.7092525 -0.8200653 -0.708678 0.335994 -0.3626041 0.4222365 -0.86997795 0.38407466 -0.5754893 0.512052 -0.9628057 0.276659 0.68981034 -0.73327523 -0.19405417 -0.7581972 0.36514184 -0.13675752 -0.9363435 0.42766616 -0.5805686 -0.27735272 0.1190573 0.33971685 0.17694983 -0.82430166 0.13912335 0.082844935 -0.9310221 0.13415422 0.95909625 -0.8462478 0.10641549 -0.4245745 -0.98424494 1.1939193 0.31622794 -0.80078983 0.2840097 0.69884175 0.33123273 -0.10990433 0.0820752 -0.11238371 0.9017404 -0.74221087 -0.5890807 -0.0975937 0.6988849 0.6424787 0.48282883 -0.53648895 0.74825907 0.74925226 -0.029674448 -0.7072113 -0.97818035 0.6602806 -0.32843366 -0.63957846 -0.92119116 0.7490152 0.7113028; 0.9560235 -0.9786159 -0.8105212 0.15962982 0.9961003 -0.8570374 -0.8557079 0.41166645 -0.68452066 -0.31024012 0.022939706 -0.5416607 0.55993485 -0.26195854 -0.4079469 0.71315455 -0.19083051 -0.410997 -0.6353264 -0.6952828 0.030178914 0.20195141 -0.018870253 -0.5957949 0.8801823 -0.7266171 0.37190443 -0.6683654 0.61814374 0.7790044 -0.791068 -0.24874313 -0.64493275 -0.27217537 -0.05882741 -0.98611456 0.46033713 -0.4556922 -0.21505177 0.0070353337 0.2718391 0.046886314 -0.942528 0.041777182 0.094874926 -0.81819785 -0.18176986 0.80922765 -1.0018578 0.31515115 -0.79944867 -0.6182437 0.668425 0.26915812 -0.8208087 0.46068686 0.968122 -0.107629076 -0.69822335 0.4544529 0.27652773 0.5354378 -0.15665081 -0.9564388 -0.2699796 0.9826442 0.9502784 0.62322634 -0.8804095 0.92366755 0.78594196 0.28207758 -0.9429853 -0.76545095 0.24513401 0.046363056 -0.40093395 -0.8384577 0.75588626 0.5388534; 0.70896405 -0.8411355 -0.8665108 -0.29581916 0.8436813 -0.7768355 -0.8455428 0.5016718 -0.685105 -0.39356053 0.27435446 -0.6968992 0.78757936 -0.4111146 -0.57333153 0.7678593 0.118392505 -0.46051738 -0.71974194 -0.75375783 0.1740691 -0.12560683 0.2652437 -0.08865612 0.7646804 -0.9001672 -0.0041126953 -0.70947 0.25230598 0.8535094 -0.8083461 -0.34379458 -0.5602198 0.18463433 -0.56425065 -0.7263247 0.08107957 -0.6469694 -0.2806019 -0.3655568 0.24630743 -0.35874364 -0.9015884 -0.23212479 0.30052173 -0.74424994 -0.0038326795 0.5787091 -0.8408849 -0.22059564 -0.8857081 -0.6226571 0.6555038 0.4661161 -0.82708067 0.22936806 0.78149354 -0.49123085 -0.5230537 0.3006252 0.3712655 0.7042402 -0.48000234 -0.73988074 -0.414034 0.89170533 0.7269282 0.87999904 -1.0300612 0.9955591 0.84771305 -0.12699282 -0.7050749 -0.8270175 0.22808996 0.085599855 -0.44106138 -0.7130156 0.6178408 0.61119527; 0.54216933 -0.83467567 -0.5841183 0.39652303 0.7115888 -0.7039678 -0.66552943 0.6680378 -0.87226075 -0.43195578 -0.36144295 -0.8299646 0.49229375 0.3977887 -0.40724966 0.8325591 -0.5621344 0.13700259 -0.6474957 -0.76400155 -0.20567536 0.6182082 -0.3779486 -0.58634067 0.97470737 -0.65360695 0.41871548 -0.43616444 0.43635726 0.63687044 -0.90869987 -0.10573378 -0.7312168 -0.28084528 0.455539 -0.68775964 0.6984244 -0.6212591 0.35339263 0.58815336 -0.18135415 0.29287162 -0.77153766 0.22595173 -0.5401865 -0.5731995 -0.34918654 0.9727253 -0.7800331 0.45720023 -0.7692935 -0.41053724 0.6834089 -0.02414733 -0.4611289 0.5855606 0.55591124 0.27750507 -0.5603137 -0.39730573 -0.45932674 0.48365504 0.03432314 -0.5875627 0.2364614 0.46949616 0.79325813 0.6492367 -0.48128256 0.8343339 0.77348304 0.71384066 -0.77981657 -0.6306942 0.37985072 -0.48485336 -0.24634725 -0.608595 0.48161852 0.609949; -0.8715212 0.7809427 0.8205264 -0.51368463 -0.7670354 0.79701066 0.8701449 -0.6454382 0.8935943 0.5525015 0.52178884 0.56527114 -0.6785379 -0.21778722 0.22490305 -0.8199965 0.41653302 -0.21500164 0.62755394 0.9070988 -0.016346192 -0.33029675 0.18325694 0.67020416 -0.9344424 0.5876588 -0.5251658 0.4174848 -0.5110577 -0.58906204 0.72922635 0.29122648 0.65038687 0.24174362 -0.07271194 0.6891163 -0.49180338 0.5796133 -0.25703833 -0.51613027 0.17865188 -0.10165805 0.8376786 -0.14412992 0.34733784 0.82289803 0.5488617 -0.7414854 0.7947629 -0.31105295 0.72284317 0.53643805 -0.9221786 -0.21492651 0.7964095 -0.47563478 -0.8089682 -0.4995769 0.39760762 0.12285877 0.09466068 -0.43181342 -0.0032633182 0.62419385 -0.18026376 -0.8553675 -0.530735 -0.45446184 0.72714585 -0.94041955 -0.78844726 -0.6312419 0.97498167 0.7053963 -0.6921761 0.32373217 0.4020566 0.57102966 -0.62517476 -0.5499418; 0.99135214 -0.65678424 -0.8670952 0.034920234 0.90430486 -0.66756135 -0.72433275 0.669374 -0.95852226 -0.34883994 0.1072871 -0.59366095 0.7389478 -0.34811327 -0.6661797 0.8905843 -0.09731507 -0.41843528 -0.7106362 -0.84864485 0.37552536 -0.031815864 0.13709334 -0.33855292 0.67449224 -0.8174614 -0.19685063 -0.7793553 0.1211996 0.67229706 -0.6082528 -0.51594853 -0.25877032 0.048545294 -0.5366191 -0.7803323 -0.14249422 -0.8537028 -0.33668494 -0.20443392 0.4702222 -0.43268985 -0.6228529 -0.4179441 0.14967354 -0.796195 -0.106100954 0.71725947 -0.667958 0.056592677 -0.8776654 -0.7244557 0.66429096 0.5643123 -0.70457834 0.0543194 0.95824593 -0.26272807 -0.3101963 0.44067103 0.4684003 0.8404392 -0.5186454 -0.8435893 -0.23341861 0.901959 1.0461177 0.5492241 -0.9043428 0.6038824 0.9891929 -0.009967975 -0.9216209 -0.6267311 0.016568989 -0.1343155 -0.48049054 -1.0586398 0.77396774 0.7783381; -0.23345977 0.44116366 0.44964102 0.6221692 -0.23967898 0.27928475 0.44319323 -0.41646472 0.46067902 -0.57056534 -0.6161679 0.33686715 -0.40780124 0.38061997 0.62872064 -0.59075177 -0.7932507 0.58917755 0.51087433 0.3203573 -0.5644081 0.6136975 -0.5082491 -1.0534244 -0.72474873 0.38545567 0.70350647 -0.54707897 0.5821254 -0.65884143 0.44382071 0.68968225 -0.8071183 -0.4420034 0.94913024 0.4028456 0.7582811 -0.5245705 0.71919227 0.87689155 -0.7311564 0.64000815 0.3017883 0.7285353 -0.5107613 0.44384113 -0.5244486 -0.22798476 0.4803148 0.7307083 0.61507833 0.07771423 -0.2892722 -0.84076095 0.3870713 0.6953026 -0.65828687 0.75169146 0.620781 -0.5630665 -0.7881524 -0.20896268 0.32705966 0.38370806 0.73806405 -0.4942425 -0.6450867 -0.007032273 0.53362393 -0.36353534 -0.46799982 0.6708639 0.40072998 0.34821624 0.85623246 -0.6826023 0.4692678 0.3895344 -0.08081662 -0.29620504; -0.080253266 0.3976051 -0.15748581 0.91615313 -0.23629092 0.57290596 0.19243418 -0.44769984 0.38478285 -0.6633208 -0.8095996 0.20721573 0.16958627 0.6331264 0.70504946 -0.36326933 -0.988584 0.6383632 0.43538243 0.36760893 -0.48230627 0.74830866 -0.46288902 -1.034553 -0.36671218 0.2424579 0.86040336 -0.54775345 0.88317585 -0.3753468 0.22770436 0.6468233 -0.98147404 -0.84179354 0.80629367 0.25527826 1.0336703 -0.65921277 0.41211328 0.89145553 -0.5251405 0.7544142 0.4433055 1.0066313 -0.5468973 0.17246309 -0.6943049 -0.50148827 0.4062769 0.6644221 0.4567735 -0.38831168 -0.04679687 -0.76898456 -0.20221794 0.86430717 -0.07087339 0.7619347 0.5346809 -0.9707187 -0.77141726 0.26967257 0.6013783 0.5072346 0.7066153 -0.32093662 -0.45683035 0.22165579 0.03348402 -0.49386984 -0.34675834 0.7764906 0.28514728 0.42440453 1.0682656 -0.9060252 0.62630385 0.012774843 0.4043841 0.18420307; -0.9882492 0.97367615 0.8313014 -0.06376345 -0.7810442 0.8021792 0.7758515 -0.49686918 0.8086436 0.5796147 -0.37109116 0.8386058 -0.71996963 0.26318356 0.55106694 -0.42875084 0.17498155 0.5615998 0.83248353 0.8614375 -0.47337884 0.021360828 -0.31989348 0.28694832 -0.8134769 0.9067765 -0.035375405 0.56545556 -0.26533943 -0.9706004 0.64761645 0.49695376 0.65511763 -0.015291674 0.126323 0.802392 -0.21978405 0.34505212 0.34275132 0.22491473 -0.23559825 0.08979313 0.8708911 0.13476752 -0.1577276 0.9986685 0.13926539 -0.9190835 0.74142534 -0.13702507 0.8208176 0.51233417 -0.7581607 -0.15213868 0.77855766 -0.21007448 -0.80934286 -0.14133382 0.7204423 -0.53191686 -0.03316176 -0.607053 0.59115046 0.7982305 0.25093523 -0.86906284 -0.95570743 -0.70304924 0.6826363 -1.0226144 -0.8861579 -0.11380109 0.8370438 0.8476861 -0.41094032 -0.0042931587 0.33881524 0.73547107 -0.614303 -0.66163915; 0.44764605 -0.47505695 0.3712553 -0.85240203 0.56204504 -0.6173999 -0.28662607 0.22789238 -0.29900065 0.9815699 0.71856886 -0.6288844 -0.35232338 -0.94905585 -0.60664666 0.2694485 0.5038192 -0.84724283 -0.5372554 -0.52425206 0.8705408 -0.6651017 0.84534866 0.8244066 0.13402386 -0.04419596 -0.50544983 0.35133573 -0.89496195 0.34723502 -0.46621358 -0.6429805 0.49426007 1.0326409 -0.6566124 -0.46052068 -0.53355145 0.6207043 -0.86960506 -0.6572472 0.72653973 -0.7119274 -0.2891879 -0.65899867 0.99016154 -0.7090306 1.0750781 0.36649382 -0.29639688 -0.6300669 0.04635966 -0.0595072 0.5461701 0.5232926 0.26340643 -0.55959636 -0.06743467 -0.65052676 -0.42282093 0.80697465 0.62310475 0.17743108 -0.7910227 -0.0854368 -0.51538134 0.063723095 -0.003284643 -0.9906847 0.007760716 0.49829364 0.39465877 -1.0119398 -0.35799474 -0.2517911 -0.8226289 0.9938393 -0.49139374 -0.19915241 -0.052386045 0.1480446; 0.1790156 0.3144869 -0.23591603 0.8501783 -0.049330845 0.4441787 0.14621605 -0.4168374 0.15947542 -0.5819333 -0.7809635 0.4507308 0.33567527 0.61131114 0.5210316 -0.61148983 -0.8317233 0.73800635 0.24605125 0.33154616 -0.8239533 0.7563745 -0.87058747 -0.9130062 -0.3975075 -0.0810576 1.0505829 -0.8255901 0.7712708 -0.26535746 0.41955674 0.6611959 -0.7474746 -0.6916514 0.76007015 0.20711182 1.0035655 -0.53643733 0.4753763 0.6475037 -0.7949951 0.7165494 0.110482246 0.9443155 -0.7235149 0.16512135 -0.69693124 -0.41177323 0.020875733 0.7046373 0.22910258 -0.6352252 -0.29641598 -0.6434348 -0.32554406 0.98780566 0.29935244 0.7686452 0.40731567 -0.5695171 -0.6843754 0.35158536 0.697025 0.10952886 0.83396655 -0.012917248 -0.3519491 0.5196187 0.05470042 -0.3999419 -0.19745068 0.80865407 0.25893557 0.20302007 0.91141987 -0.7176038 0.49783495 0.24669948 0.30267215 0.31311935; -0.64730185 0.8611157 1.0303955 0.14460674 -0.84057695 0.78070134 0.802779 -0.60695714 0.9288832 0.118137166 -0.3406963 0.56506944 -0.8458262 0.40566084 0.3567374 -0.8663714 -0.22816986 0.54905164 0.7307054 0.59604436 -0.41785908 0.3957942 -0.24566069 -0.010339106 -0.6387962 0.6666549 0.331572 0.6992356 0.13462925 -0.9062884 0.685068 0.6623925 0.16527376 -0.31243652 0.5514058 0.76907 0.33398852 0.48453414 0.49925673 0.4955229 -0.5109703 0.5208396 0.9020079 0.2776913 -0.13909604 0.6687948 -0.115405545 -0.6050174 0.63071376 0.29437977 1.0133592 0.5149566 -0.77993 -0.6968013 0.8373617 0.22962715 -0.8515706 0.37088487 0.44063735 -0.39502734 -0.60912067 -0.9091787 0.25594133 0.70344543 0.63410777 -0.9235614 -0.7666357 -0.7257258 0.9134703 -0.77676153 -0.75943077 0.08404009 0.8582321 0.9389243 0.33796588 -0.2502766 0.47209334 0.8514211 -0.84302074 -0.5717542; 0.24195589 -0.7716055 -0.6382104 0.5944528 0.31214282 -0.6313193 -0.25187513 -0.33840212 -0.83865446 -0.54194367 -0.8771737 -0.057928514 0.6713837 0.6285802 0.8563246 1.1132392 -0.45132926 1.0253674 -0.45451784 -0.59917116 -0.87244695 0.82526296 -1.0542676 -0.28219292 0.9410167 -0.48973924 0.40400627 -0.15948273 0.6546143 0.8258313 -0.021262992 0.30716524 -0.15474457 -0.79193413 0.52977145 -0.31250742 0.448425 -0.39293575 0.84759605 0.7360721 -0.7796067 0.21768247 -0.66104496 0.4950312 -0.5847129 -0.37330627 -0.9157743 0.50118047 -0.3614381 0.9131852 -0.7294145 -0.1832281 0.39637506 -0.528092 -0.565715 0.37684757 1.0471296 0.35112497 -0.031071797 -0.5881857 -0.28138134 0.34514368 1.133758 -0.8365838 0.42694893 0.5740118 1.0234334 0.79506284 -1.0567181 0.59244275 0.8630098 0.9002891 -0.92936623 -0.51538664 0.49627092 -0.48525316 1.4619147 -0.4129542 0.1287247 0.24384508; 0.039973408 0.15630144 -0.5332287 0.747476 -0.29866457 0.19610697 -0.093990974 -0.3546355 0.09039032 -0.50141835 -0.62667906 0.18470268 0.37134403 0.6624102 0.8615587 -0.50157934 -0.83413494 0.5991445 0.2731174 0.43824667 -0.6300602 0.6056196 -0.63910294 -0.72930664 -0.3235043 -0.031801328 1.0968683 -0.47607973 0.8104967 -0.14889866 0.1746501 0.77665246 -0.845316 -0.93794155 0.90731806 0.078994684 0.73090446 -0.7394519 0.8667182 0.680001 -0.5851914 1.0900264 -0.00077350053 0.99947983 -0.7382289 0.19309615 -0.67797893 -0.4017547 -0.06884727 0.9734976 0.13092937 -0.54695445 -0.16166157 -0.7814113 -0.5721596 0.7568917 0.21609415 1.0509787 0.48719576 -0.6480203 -0.6922956 0.54856706 0.4944459 0.042316392 0.7306485 0.05574401 -0.23463042 0.5969452 -0.14100054 0.054899 -0.4636023 0.81807244 0.038469754 -0.049251553 0.82978725 -0.9489107 0.5939829 0.1934995 0.7190089 0.4439479; 0.088234454 0.08229049 -0.30664113 0.8975246 -0.048242725 0.25596073 0.043903958 -0.17446509 -0.0380314 -0.66399187 -0.8456053 0.3227666 0.45582902 0.71693486 0.8340001 -0.42739928 -1.0013617 0.59377456 0.17760082 0.1675671 -0.7592643 0.7358222 -0.6817891 -0.98000354 -0.45889902 0.051685378 0.9844394 -0.6323124 0.77546763 -0.29962936 0.14140697 0.9682009 -0.66856074 -0.9577732 0.7963286 -0.26061043 0.83175224 -0.5248371 0.62657 0.7631731 -0.7614812 0.8663548 0.034479976 1.0122323 -0.991441 0.25444242 -0.5766693 -0.2572587 0.07885097 0.80931294 -0.03174948 -0.443725 0.03704402 -0.6635422 -0.35131344 0.81108433 0.44792375 0.68123287 0.7158688 -0.64497393 -0.9083756 0.56960565 0.7477995 -0.12304835 1.0283542 0.23757653 -0.13903068 0.35378373 -0.098753266 -0.21727814 -0.25986657 0.73533213 0.11204558 -0.11436585 0.7962992 -0.7473676 0.47339857 -0.2048854 0.40753126 0.44899207; 0.03128177 -0.3446448 0.516087 -0.7198265 0.090638496 -0.42058918 -0.0138663985 0.63420534 -0.010105716 0.7566971 0.839154 -0.23977391 -0.3044649 -0.7184909 -0.8484192 0.40141746 0.78834516 -0.6807737 -0.16937327 -0.37507963 0.9147063 -0.91700584 0.63235545 0.91380465 0.5295383 0.06979845 -0.929955 0.49833426 -0.6806091 0.16204554 -0.4446554 -0.6196452 0.72248566 0.93154466 -0.649155 0.016554479 -1.0933901 0.5156867 -0.61657345 -0.6594723 0.5536842 -0.9543763 -0.041095022 -0.67463964 0.68846786 -0.2711934 0.9171759 0.39338768 -0.019719828 -0.78256726 0.06982344 0.6297798 0.20240363 0.90189576 0.22391553 -1.0110161 -0.27795273 -1.1902701 -0.5384737 0.9277679 0.62236124 -0.33273205 -0.5718632 -0.30003682 -0.7635491 0.09341935 -0.0028881037 -0.55442435 0.10769133 0.26519653 0.49466792 -0.53970975 -0.4562446 0.0881779 -0.73743963 1.0608538 -0.68673664 -0.1050456 -0.6147815 -0.60920984; 0.09641788 -0.5751101 0.15146895 -0.78094536 0.56927186 -0.49507654 -0.21367781 0.47153527 0.06550269 0.84737396 0.8942198 -0.4460782 -0.31750858 -0.7893186 -0.48896328 0.35913664 0.63155967 -0.9319024 -0.23881894 -0.52349466 0.65300936 -0.8121471 0.88602996 0.578823 0.15078364 -0.108528785 -0.6741259 0.4557557 -0.7022504 0.3791693 -0.5056053 -0.49786168 0.44639072 0.8953268 -0.91978717 -0.28199968 -0.6915275 0.5817205 -0.97023994 -1.1941206 0.9270266 -0.56919914 -0.38314497 -0.7125979 0.81864214 -0.32955313 1.0271168 0.22597127 -0.46366546 -1.0115827 0.25543928 0.096812665 0.07073698 0.77452505 0.26129758 -0.83583975 -0.2899532 -0.76535517 -0.48837107 0.87917703 0.63856995 -0.24758922 -0.9002204 -0.051983807 -0.6781383 0.19334313 -0.17481275 -0.58637667 0.20885226 0.38818035 0.040324587 -1.0340381 -0.3285934 -0.34376407 -0.65619767 1.0305258 -0.539777 -0.16864525 -0.10343295 -0.40182656; 0.51476824 -0.7513969 -0.7883706 -0.86829734 0.4427329 -0.46121582 -0.44636235 0.2705554 -0.64704794 0.8007323 0.43504682 -0.50009847 0.65071225 -0.55219734 -0.30623323 0.76426184 0.5707986 -0.4481347 -0.2725465 -0.19835353 0.5135693 -0.43679887 0.42386398 0.60373235 0.6998998 -0.6165829 -0.6943067 0.41870633 -0.6141649 0.737152 -0.4671937 -0.89400524 0.60953933 0.37929103 -0.8280535 -0.2814032 -0.67135364 0.6980162 -0.56985897 -0.81079197 0.2903176 -0.79680616 -0.8253301 -0.6965022 0.79676324 -0.19910501 0.66259784 0.48336914 -0.29811186 -0.27764335 -0.48321092 -0.090998605 0.17092848 0.6610751 -0.54116166 -0.53160465 0.39652616 -0.69734126 -0.4667375 0.9123132 0.67716426 0.3920654 -0.71083385 -0.54137623 -0.9218422 0.5698406 0.5771715 -0.007921032 -0.69941556 0.81458783 0.69826543 -0.6851943 -0.49839577 -0.69871455 -0.7042217 0.92963356 -0.1707954 -0.460497 0.42920718 0.39442608; 0.9163477 -0.8646289 -0.9977549 -0.39149097 0.60975593 -0.8049676 -0.79975635 0.55790097 -0.9775585 -0.08875503 0.17615752 -0.7939529 0.9628542 -0.2880259 -0.4750903 0.7519994 0.4083723 -0.46287242 -0.7467165 -0.8892721 0.37193123 -0.2796441 0.4681622 -0.10691586 0.5321666 -0.61596775 -0.29881102 -0.40120926 -0.24684419 0.7764983 -0.46407518 -0.35383078 -0.26497212 0.22168417 -0.35138842 -0.7957472 -0.12506686 -0.67108315 -0.45719317 -0.5560511 0.32482225 -0.44101605 -0.7739688 -0.49625716 0.4704359 -0.7414966 0.13029914 0.5868313 -0.8763188 -0.4001644 -0.64478415 -0.25070056 0.7927044 0.42797813 -1.0181879 -0.17150149 1.099387 -0.46374276 -0.3151458 0.49609923 0.5583527 0.60546124 -0.1903823 -0.9534831 -0.5428558 0.9631501 0.70223105 0.97167134 -0.82515985 0.7735462 0.6812916 -0.0814066 -0.7956357 -0.8985043 -0.29595104 0.20588824 -0.56098163 -0.5791683 0.64405745 0.6397177; 0.87171495 -0.8666817 -0.55751586 0.16508575 0.6497856 -0.91466945 -0.90498465 0.5573501 -0.977595 -0.48207158 -0.13829651 -0.5971442 0.49912035 0.06949551 -0.5662996 0.9185478 -0.22712581 -0.030554073 -0.63131005 -0.92522055 -0.010552158 0.15639956 -0.24104334 -0.32178137 0.85704273 -0.6670501 0.28456622 -0.44595975 0.56383085 0.9239413 -0.67735565 -0.378029 -0.4264196 -0.39779332 -0.0048935204 -0.91103673 0.24180123 -0.5644757 -0.22056372 0.047899954 0.15245922 -0.16385235 -0.78866607 -0.035249814 -0.4268131 -0.7333801 -0.4533355 0.66705894 -0.6191245 0.2664087 -0.7599084 -0.36634177 0.66962785 0.1233582 -0.73124456 0.3363696 0.76187235 0.18044001 -0.35292274 0.18778814 -0.07071451 0.49384278 -0.34829414 -0.68296075 -0.045239076 0.775443 0.75107926 0.63450104 -0.66738653 0.72201794 0.9501319 0.47661147 -0.7938682 -0.74888897 0.25207144 -0.3511868 -0.29835683 -0.8251762 0.43847647 0.6007032; -0.8370591 0.79398876 0.622986 0.17223462 -0.7538077 0.56234634 0.57361865 -0.34106103 0.6920788 -0.015936894 -0.4155813 0.69208294 -0.668848 0.22474957 0.3089884 -0.5005803 -0.28155184 0.34408274 0.72458506 0.5894367 -0.62444085 0.42782217 -0.5067644 -0.16355807 -0.80450463 0.80204314 0.50830257 0.55678 0.17484789 -0.78400785 0.4885299 0.531953 -0.2130333 -0.49615806 0.5193144 0.82976145 0.4483029 0.7073076 0.24616803 0.4975632 -0.5927726 0.30256987 0.7599832 0.43275294 -0.34712574 0.7957645 -0.18908742 -0.6548878 0.6480143 0.24361685 0.66335815 0.7704617 -0.7379968 -0.57677454 0.96520656 0.47461388 -0.86745334 0.5359309 0.6090386 -0.45818487 -0.54617196 -0.8962023 0.5905252 0.61168283 0.523161 -0.7459816 -0.8655241 -0.9404014 0.636191 -0.71236575 -0.6766566 0.3159688 0.80655694 0.9076313 0.49880436 -0.41847396 0.294321 0.63969046 -0.726369 -0.8223357; 0.7044558 -0.58740336 -0.49748155 -0.29665637 0.59000576 -0.39672405 -0.79358584 0.42438138 -0.8615188 0.28198925 0.46408835 -0.47869888 0.58823985 -0.54243064 -0.6047536 0.58494914 0.3641415 -0.58544695 -0.6323428 -0.36250678 0.4390049 -0.4786224 0.4018731 0.39411142 0.515423 -0.6921767 -0.35865894 -0.47355253 -0.5690868 0.5191202 -0.5995516 -0.6388332 0.4167284 0.3378434 -0.59213984 -0.6229714 -0.4300674 -0.7044213 -0.2558863 -0.5198931 0.36977884 -0.5672457 -0.6503251 -0.49474204 0.21509482 -0.52521306 0.4551006 0.3315919 -0.44415486 -0.37589556 -0.75549555 -0.6354137 0.5300292 0.31465122 -0.7085138 -0.50819755 0.67537665 -0.62763333 -0.4428182 0.5718118 0.5263681 0.46977723 -0.54819167 -0.82358104 -0.49837714 0.63448787 0.5964068 0.17922495 -0.5426552 0.53166777 0.8572595 -0.52241576 -0.55048853 -0.6328544 -0.42250746 0.5524288 -0.54738075 -0.75766 0.48465794 0.779768; 0.40477034 0.11757307 -0.3130637 0.9720744 -0.1374952 0.5592361 -0.048211087 -0.5821674 0.081750885 -0.8148621 -0.5632844 0.48820028 0.58374447 0.7423966 0.8678193 -0.5257241 -0.8465986 0.82183397 0.17432722 0.36321923 -0.9117453 0.7581912 -0.75009364 -0.74413776 -0.60850614 -0.18903477 1.0128893 -0.8271493 0.87376255 -0.09354227 0.19364488 0.95970106 -0.67511564 -0.8194461 1.085531 -0.22210832 0.65265906 -0.58044386 0.53201777 0.88402385 -0.8366653 0.9538117 0.103353746 0.8632017 -0.7662533 0.025156591 -0.87935907 -0.30643526 0.036357943 0.94006133 -0.070670515 -0.49480444 0.023167472 -0.98160464 -0.5889694 0.6797929 0.2242002 0.6821604 0.71144557 -0.88622344 -0.934672 0.70359546 0.509261 0.12730804 0.92726094 0.20271918 -0.24732742 0.43150824 -0.11587943 -0.22761177 -0.42003575 0.78472054 -0.030083336 0.017859207 0.87368464 -0.72134537 0.71923214 -0.03680155 0.7209279 0.4852657; 0.7810784 -0.8935103 -0.5668823 -0.16625406 0.97053903 -0.59400123 -0.75065315 0.48949572 -0.81116796 -0.2873454 0.35733256 -0.68957406 0.85074663 -0.47198597 -0.33788663 0.43898383 0.082150035 -0.7037843 -0.79486436 -0.58147156 0.5499136 -0.3027375 0.5971575 -0.20554797 0.4404394 -0.6090937 0.0080807 -0.7857637 -0.16969047 0.47687697 -0.7867583 -0.22874671 -0.50868577 0.33146414 -0.14332786 -1.0178045 0.21475081 -0.53073674 -0.40414536 -0.14778145 0.49297175 -0.25487265 -0.55613583 -0.24405052 0.52010727 -1.0245318 0.3429405 0.6063641 -0.72198033 -0.09506037 -0.6522136 -1.0865736 0.75249547 0.14286557 -0.73857284 0.007828049 0.6999004 0.044854175 -0.4829804 0.2324896 0.4053833 0.7541051 -0.5193943 -0.5263516 -0.3524163 0.769888 0.62222946 0.4710733 -0.85714287 0.5638026 0.62384725 -0.3769742 -0.7212961 -0.64195585 0.11576752 0.04864275 -0.29106444 -0.86218214 1.0187746 0.91153514; 0.8546248 -0.6616252 -0.68957597 0.44897887 0.6982913 -0.70404446 -0.62915695 0.7952558 -0.52982134 -0.71352166 -0.41543794 -0.64856327 0.7667347 0.15815707 -0.13509306 1.0092509 -0.49828342 0.31814244 -0.62557644 -0.7752064 -0.31915876 0.4430837 -0.5265063 -0.4281438 0.8991898 -0.6948194 0.6791322 -0.5540051 0.53060186 0.56725645 -0.8206213 0.2509677 -0.51080394 -0.5111397 0.18550049 -0.55083996 0.75601 -0.44146138 0.22896235 0.41068992 -0.058744412 0.5981995 -0.59969383 0.22374012 -0.5492586 -0.8262146 -0.48387393 0.67147094 -0.6509405 0.673754 -0.61366063 -0.64848566 0.9455216 -0.020750795 -0.4262501 0.63834023 0.7163661 0.3408397 -0.24105789 -0.257589 -0.48793676 0.7572385 -0.014998636 -0.6750211 0.30612993 0.69579935 0.7904309 0.48274294 -0.52771914 0.612269 0.67154974 0.45754728 -0.837056 -0.816222 0.6392775 -0.56687415 -0.223537 -0.7707138 0.7904293 0.61053485; 0.76315415 -0.6214475 -0.9575493 -0.4910604 0.8334401 -0.76958144 -0.7448257 0.52836084 -0.6846064 -0.1840377 0.45753643 -0.58608705 0.80358756 -0.49462676 -0.56099254 0.83974934 0.2564775 -0.25284404 -0.8471321 -0.57245076 0.2602893 -0.24090058 0.34432775 0.056580957 0.6795319 -0.8525141 -0.1945499 -0.5623605 0.029409256 0.6548147 -0.5177319 -0.59638727 -0.2830022 0.38078976 -0.38038737 -0.87056994 -0.23740079 -0.73961055 -0.49549308 -0.59172165 0.4273658 -0.5822589 -0.73413616 -0.27879503 0.3978759 -0.49250832 0.3140218 0.6145977 -0.76016724 -0.14347188 -0.89626265 -0.60122794 0.6215 0.67214507 -1.0124091 -0.10773715 1.0014775 -0.4357267 -0.5966927 0.6014687 0.4569928 0.96525943 -0.4994006 -0.6682374 -0.28046036 0.81137145 1.0321649 0.9978068 -0.81090623 0.8687082 0.88271874 0.00061680796 -0.98554254 -0.98940253 -0.3877553 0.41493696 -0.39306816 -0.69845426 0.5973414 0.8381034; 0.3061556 -0.1785572 -0.727898 0.9635455 -0.17492501 0.2543029 -0.10829831 -0.57987666 -0.69466996 -0.6148244 -0.9545252 0.53407276 0.4584982 1.1347386 0.5871466 0.07656629 -0.56772757 0.9456496 0.21712717 0.28231183 -0.81123084 0.9158965 -1.0732269 -0.5283121 0.049606476 -0.4066198 0.6664395 -0.49243155 0.6522692 0.39508936 0.550482 0.40672982 -0.28878796 -1.0142069 0.78047675 -0.13625042 0.6420257 -0.43202522 1.1490011 0.6590683 -1.179659 0.46230286 -0.3227405 0.6694653 -0.9694132 -0.07424738 -1.0322889 -0.20869166 0.03643943 0.91660845 -0.77877486 -0.24539296 0.19574009 -0.4483589 -0.7482337 0.61876565 0.5420261 0.52902967 0.29600045 -0.48928002 -0.7218979 0.34261042 0.9001273 -0.3634263 0.62434006 0.6685489 0.62759984 0.82875985 -0.5929631 0.4296328 0.16571099 1.0977952 -0.461194 -0.46188578 0.5315328 -0.83540285 0.7689254 -0.3852894 0.293613 0.5104464; 0.12285316 -0.30636555 0.14549059 -0.78683007 0.31453946 -0.5709364 -0.186754 0.6205406 0.24094096 0.6473682 0.8607013 -0.7814785 -0.3503508 -1.0431349 -0.5539414 0.4499874 0.82776994 -0.9232702 -0.57288074 -0.50267917 0.81302506 -0.79367405 0.9317187 0.41586137 0.14439112 0.3081128 -0.5928371 0.2870879 -0.7826936 -0.12060398 -0.5940432 -0.6495689 0.41408285 1.1537923 -0.6062817 -0.36769176 -0.80426836 0.4014764 -1.0458727 -0.9021114 1.0413646 -0.7366613 -0.4798916 -0.5160326 0.8874631 -0.48383778 0.9188269 0.51960963 -0.2845597 -1.0061812 0.38765612 0.43051073 0.2171306 0.441973 0.0811431 -0.6144823 -0.2624138 -0.57878953 -0.17237602 1.0046026 0.5246672 -0.36989304 -0.90127045 0.017131543 -0.7448016 0.028673485 -0.3204903 -0.7531307 0.16515496 0.43744573 -0.08601139 -1.0384189 -0.14829391 -0.2880244 -0.6570805 0.7656814 -0.9336101 0.20312294 -0.1746617 -0.26989952; -0.73654455 0.9088576 0.61786807 0.1511104 -0.8194147 0.5838936 0.7156883 -0.8807409 0.9562875 0.42736635 -0.39735913 0.72073704 -0.87160873 0.3649848 0.27174312 -0.75696975 0.29936478 0.34082377 0.7815273 0.8857341 -0.41122746 0.33471298 -0.020886166 0.46482724 -0.605072 0.8751074 -0.22662728 0.38742006 0.017547892 -0.7822107 0.62441075 0.3111526 0.37042713 -0.088625 0.29586068 0.7052212 -0.19839264 0.815654 0.5191749 0.26170924 -0.14874837 0.34767717 0.69747883 0.4144758 0.035346903 0.60327137 -0.11282513 -0.70985585 0.7337668 -0.05316842 0.6972509 0.45329314 -0.7237474 -0.5091147 0.76620084 0.053284448 -1.0586742 0.0032450135 0.4699327 -0.43697953 -0.14239283 -0.8549925 0.19566526 0.8884557 0.3934243 -1.0267472 -1.0142709 -0.68072677 0.9407983 -0.749972 -0.92263097 -0.13585372 0.9642666 0.8855664 -0.14024383 -0.011056275 0.5052387 0.79113257 -0.71804047 -0.7984609; 0.49631867 -0.54640204 -0.2493997 -0.7534352 0.37290305 -0.5018063 -0.3465472 0.49424827 -0.426157 0.5642123 0.3897304 -0.18221314 0.44492972 -0.5738205 -0.375685 0.6651692 0.80727726 -0.44789824 -0.29092094 -0.4946215 0.61906093 -0.66652405 0.72047204 0.71606237 0.55070853 -0.612978 -0.7621249 0.50963944 -0.6819555 0.6056191 -0.3818466 -0.8635348 0.91106546 0.6153838 -0.82681876 -0.38606068 -0.6357771 0.37358665 -0.43174538 -0.6071346 0.48984998 -0.8332218 -0.3628525 -0.6456661 0.31438908 -0.36500528 0.60205865 0.38722616 -0.24599287 -0.5101285 -0.66121346 -0.25893235 0.33878556 0.71162945 -0.48524162 -0.6939598 0.7203272 -0.89144444 -0.5845069 0.60882354 0.6468911 0.34570286 -0.43164325 -0.58855534 -0.5997391 0.4022842 0.53778607 -0.37741923 -0.4354604 0.50903773 0.6929974 -0.32756016 -0.55721736 -0.5191503 -0.7784704 0.7205594 -0.3172762 -0.37720102 0.29637027 0.1566358; -1.1505706 0.710877 0.6000935 -0.28632733 -0.8960914 0.6689004 0.8658456 -0.45021847 0.62915283 0.7790053 -0.19562796 0.86312276 -0.49257225 0.2820177 0.5290165 -0.018677473 0.644563 0.43763712 0.9489934 0.63868874 -0.2343286 0.24823444 -0.16881317 0.56987613 -0.51359314 0.7040535 -0.74425405 0.81393266 -0.13241358 -0.68932605 0.97000724 0.24219719 1.0139279 -0.3338717 -0.00067717955 0.7975516 -0.69220114 0.6545686 0.3665203 -0.35141182 -0.35743675 -0.93867755 0.6172926 0.06417632 -0.0011493425 0.9353534 -0.2381564 -0.74802375 1.0113598 -0.035226673 0.67208433 1.41108 -0.8117776 0.09363508 0.39936176 -0.23650979 -0.050222643 -0.98561513 0.33734372 -0.03581263 0.04121378 -0.87195385 0.62113535 0.6455308 0.021883942 -0.36777064 -0.64347905 -0.07138553 0.27653405 -0.71195453 -0.60500467 0.24166343 0.42330024 0.94966376 -0.6161695 0.67013997 0.3089496 0.7701855 -1.2110913 -0.9371966; -0.027517904 -0.19772892 0.32080442 -0.8543536 0.23621196 -0.44003376 -0.13190554 0.33871657 -0.3538653 0.8251896 0.84970355 -0.3491995 -0.6031447 -0.45046493 -0.83030176 0.553332 0.8227854 -0.69424754 -0.048366535 -0.31375778 0.8728333 -0.764228 0.7859262 0.81156534 0.48485956 -0.21392035 -1.0610155 0.5136377 -1.0205116 0.23054129 -0.15517138 -0.814178 0.860298 0.6519885 -0.9109614 -0.111204855 -0.7975712 0.6134752 -0.44590572 -1.0704948 0.69132644 -0.7911265 -0.30687574 -0.8931016 0.65880275 -0.20921144 0.89721435 0.27109253 -0.28915548 -0.89869523 -0.3865524 0.4737697 0.11051599 0.9459657 0.2668386 -0.9663825 -0.110942245 -0.6167941 -0.73071885 1.0356698 0.77215534 -0.32286304 -0.5089725 -0.20306587 -1.0032183 -0.045619454 0.51572555 -0.27635542 -0.06749444 0.22564967 0.4021105 -0.8838448 -0.33600378 -0.11159806 -0.8006447 0.8548147 -0.8101274 -0.08831475 -0.50378966 -0.5584894; 0.785819 -0.73130476 -0.46096984 0.52695674 0.6194621 -0.97085816 -0.5843052 0.77474725 -0.8325294 -0.41740972 -0.38101062 -0.7581303 0.77653116 0.04844479 -0.19474488 0.9205558 -0.61341614 0.16204654 -0.6959981 -0.986086 -0.2802353 0.32266042 -0.5164925 -0.7355049 0.77347606 -0.5123868 0.34043002 -0.680232 0.6359253 0.89319605 -0.89697015 -0.18232253 -0.579792 -0.557462 0.11425906 -0.77705145 0.68138015 -0.71349764 0.2437675 0.39302897 -0.05699231 0.37720346 -0.9784963 0.4052593 -0.5087434 -0.9603879 -0.66489613 0.8541993 -0.7959841 0.3472042 -0.809229 -0.5786134 0.6186331 0.096493036 -0.68302155 0.43108782 0.60563785 0.29003212 -0.8572704 -0.25639242 -0.14248323 0.46342963 -0.07693553 -0.71019566 0.40149495 0.6161773 0.50538415 0.7905325 -0.813013 0.6686884 0.8213031 0.45232123 -0.77888334 -0.6296113 0.72582763 -0.6144972 -0.39164764 -0.69368374 0.6071781 0.73854876; -0.4574154 0.2892561 -0.3776611 0.7358684 -0.67133844 0.56511664 0.5532177 -0.4109701 0.20987451 -0.9038862 -0.80856556 0.48920488 0.27193558 0.9572816 0.5708796 -0.13723941 -0.66370994 0.9601612 0.642756 0.4414677 -0.9020824 0.8807324 -0.74826545 -0.5932314 -0.3926051 -0.009033006 0.82192194 -0.3582795 0.8963742 -0.38709787 0.5951643 0.38639674 -0.6356484 -1.074882 0.43079588 0.31559598 0.54939896 -0.773676 0.87499666 0.7024415 -0.8146687 0.37651363 0.43648657 0.69110435 -0.8188624 0.5099656 -1.0383945 -0.63948846 0.38002402 0.9343531 0.10062305 -0.020571485 -0.27862552 -0.3787045 -0.28935602 0.7710938 0.018988555 0.35374713 0.13083512 -0.45698953 -0.76249456 -0.031106493 0.86603075 0.043463152 0.6555607 -0.1935371 0.06697972 0.6559058 -0.0072809523 -0.17540906 -0.18470779 0.9887749 0.26308724 0.21193479 0.80579734 -0.9316612 0.8323707 0.25343925 0.099617854 0.19738996; 0.14451726 -0.28385043 0.40202245 -0.5240379 0.25912562 -0.57584655 -0.21284902 0.54382294 -0.44023338 0.5530562 0.78474647 -0.30807963 -0.4380813 -0.61340874 -0.44424963 0.41713086 0.8108268 -0.65673184 -0.3150972 -0.30466467 0.688562 -0.83796966 0.6892755 0.8943363 0.4560742 -0.12936224 -0.782605 0.69557023 -0.5795366 0.35044184 -0.36465895 -0.79528874 0.57308376 0.8008746 -0.86655074 -0.23585565 -0.98087645 0.6247426 -0.610547 -1.1037116 0.4628388 -0.8897775 -0.37752056 -1.0146332 0.43956876 -0.34973243 0.68812233 0.17808555 -0.3152729 -0.8435177 -0.18350127 0.21966551 0.24907492 0.74795663 0.29476017 -0.7912699 -0.15701844 -0.77943134 -0.65631235 0.70943797 0.74025214 -0.28388035 -0.5165755 -0.4619879 -0.7612907 0.009401124 0.26999012 -0.42865813 0.060023993 0.30141163 0.3580508 -0.8086242 -0.2394431 -0.24575128 -0.6493695 0.778856 -0.77260375 -0.237047 -0.55444175 -0.27655086; -0.6159229 0.7460838 0.79336834 -0.556427 -0.8800329 0.7340677 0.70463824 -0.464077 0.6141226 0.7445579 0.07399241 0.5699053 -0.7910242 0.06455354 0.16458035 -0.7591122 0.67259496 0.1648803 0.7808677 0.7293994 -0.023898713 -0.30256388 0.39850807 0.73069996 -0.7246215 0.83142376 -0.37042567 0.80565786 -0.3526175 -0.81610644 0.97868335 0.3521223 0.7291064 0.29633155 -0.09106098 0.90001243 -0.52624035 0.76957417 0.114159696 -0.18255916 -0.014323406 -0.035315935 0.67176896 -0.019517392 0.27063316 0.97315454 0.3040935 -0.6524734 0.9600627 -0.52881426 0.81910414 0.7054775 -0.7065418 -0.36600497 0.591434 -0.53134257 -0.59714234 -0.14501041 0.51000667 -0.10154408 0.19620119 -0.82362205 0.21757197 0.6655929 -0.21197486 -0.83801115 -0.5182759 -0.73439217 0.79092497 -0.9883056 -0.644272 -0.4358242 0.70575774 0.93949276 -0.41611215 0.19463581 0.46273825 0.7421034 -0.49121025 -0.8350793; 0.6593151 -0.12353348 -0.635811 0.6218879 0.35283515 0.2107343 -0.5554988 -0.27856168 -0.10802794 -0.7879387 -0.764 0.07556506 0.4228728 0.6352924 0.7810044 -0.3772988 -0.68269134 0.7243564 -0.42644927 0.07017485 -0.6004059 0.74476844 -0.49946195 -0.7219192 -0.13728294 -0.34275076 0.9741023 -0.9087422 0.7404004 0.065850556 -0.0085328305 0.904497 -0.6841957 -0.53019905 1.0195421 -0.5153476 0.68791616 -0.50587946 0.61525494 0.99778295 -0.5594337 0.88260967 -0.15030243 1.1391826 -0.546595 -0.6704272 -0.31757185 0.08641751 -0.41964948 0.61040026 -0.32324174 -0.6776998 0.538312 -0.9243124 -0.48327184 0.64890164 0.41966343 0.73905295 0.6997042 -0.8586195 -0.8721755 0.6909351 0.82889336 -0.25561866 1.0305635 0.25768253 0.25566834 0.5464012 -0.33784983 0.16291663 0.14269213 0.39996603 -0.20430599 -0.5536184 0.96362555 -0.69509006 0.5644766 -0.4897438 0.676642 0.67157614; -0.42233676 0.94416606 0.8095429 -0.39022917 -0.5331792 0.7914181 0.43827957 -0.8142767 0.90481186 0.46921533 0.57840174 0.61562294 -0.6751013 -0.45047173 0.31560108 -1.1788125 0.13252072 -0.30496135 0.67440563 0.85327494 0.12620948 -0.41159424 0.66959727 0.21836881 -1.1203896 0.9367086 -0.12702416 0.4185784 -0.57294345 -1.1615229 0.4231784 0.24432197 0.14879234 0.8457084 0.15560849 0.52938485 -0.19918512 0.43074933 -0.53815424 -0.1137363 0.48561588 -0.1669943 0.69988126 0.0260522 0.6874507 0.65245545 0.85703266 -0.8514599 0.58864164 -0.68868244 0.9673323 0.3476087 -0.49725685 -0.14730856 0.78304875 -0.4328962 -0.8603653 -0.15565981 1.0815519 -0.10040251 -0.05458887 -0.17133862 -0.18619329 0.66040385 -0.02582676 -0.9372391 -0.941049 -1.017666 0.7496165 -0.92311126 -0.9712803 -0.61495525 1.1075219 0.46378696 -0.105743006 0.5929055 0.21370834 0.69967496 -0.33499783 -0.25679937; 0.051055502 -0.14404869 0.51380175 -0.8191175 0.30705664 -0.4791823 -0.29491627 0.2845256 0.19513322 0.74888533 1.1029837 -0.52453864 -0.4565356 -0.91689026 -0.5066247 0.29893273 0.7372852 -1.0364985 -0.16184129 -0.3487127 0.718006 -0.8732992 1.109166 0.40108147 0.2010435 0.20636734 -0.6380652 0.4599892 -0.74769145 -0.09327948 -0.45914474 -0.69287705 0.3746578 0.8253687 -0.6907594 -0.25015628 -0.6792187 0.69534415 -0.98166966 -0.5751207 1.0029365 -0.44446242 -0.018998807 -0.69002366 0.8497199 -0.32845467 1.1175958 0.2914883 0.04787689 -1.019686 0.4884528 0.2769236 0.27097455 0.4590625 0.46872103 -0.63322705 -0.6256301 -0.49374253 -0.16936603 0.6436013 0.52931345 -0.29900825 -0.81565017 0.026228568 -0.5249041 -0.47994724 -0.31596953 -0.9088399 0.48418438 -0.08853174 0.20504414 -1.0571736 -0.10010783 0.12860282 -0.55239785 0.8228697 -0.63855845 0.14733523 -0.39052865 -0.50500494; 0.37694153 -0.3230827 0.061615206 -0.5721917 0.40966737 -0.5924655 -0.47189435 0.6329707 -0.42532787 0.8596497 0.75113237 -0.53495145 0.0038954 -0.7165177 -0.7754321 0.25255638 0.73712087 -0.87903583 -0.30318028 -0.29500103 0.6928304 -0.7110602 0.5734691 0.6748491 0.24234778 -0.5350097 -0.6522069 0.43141553 -0.66879463 0.201822 -0.35068056 -0.59058595 0.9279654 0.90475476 -0.55662227 -0.32251796 -0.86324865 0.7146407 -0.6548627 -0.7805389 0.5169246 -0.72592837 -0.41259402 -0.5865174 0.8289644 -0.3826733 0.8962955 0.60455227 -0.35877097 -0.64478064 -0.43529856 0.13180403 0.25406435 0.55826825 -0.216835 -0.7151816 0.1048758 -0.7223625 -0.40568742 0.7025696 0.53686315 0.0017868099 -0.8118412 -0.38014424 -0.7902939 0.2514827 0.416389 -0.37260568 -0.41072962 0.40325138 0.32931757 -0.9551314 -0.36564246 -0.3770747 -0.5822503 0.57777774 -0.40095395 -0.53288716 3.198345f-5 0.2927311; -0.29336324 -0.13129343 0.7182143 -0.77224475 -0.005237241 -0.4261841 0.23583463 0.30422345 0.12868896 0.5336395 0.6460956 -0.48137623 -0.6742321 -0.67381895 -0.7747432 0.391433 1.0083274 -0.6633524 -0.097879484 -0.18967712 0.8629317 -0.71931803 0.83181995 0.60396665 0.24928938 0.063679725 -0.71437114 0.45378545 -0.8130764 0.12976912 -0.21205723 -0.791609 0.55534357 0.8818421 -0.71443254 -0.0607458 -0.72583073 0.6386284 -0.9168282 -0.7027026 0.5517822 -0.90255547 -0.2094211 -0.7254973 0.6326005 0.03887822 0.80784845 0.17753558 0.04697404 -0.64077 -0.015059813 0.56133056 0.1969939 0.92532694 0.34933668 -0.897961 -0.34673208 -1.1788177 -0.6060411 0.5983842 0.9780421 -0.6850235 -0.7257168 -0.19101265 -0.79763526 -0.29938862 0.0020161748 -0.5267888 0.2693023 -0.11814903 0.4132567 -0.7834597 -0.12074815 0.19717862 -0.7124698 0.99398696 -0.83106446 -0.10856692 -0.5650885 -0.685854; -0.2680858 0.4126795 0.19735381 0.77618486 -0.59810066 0.564518 0.55648535 -0.38344267 0.20512052 -1.0066996 -0.81705356 0.5221399 0.2171009 0.76501095 0.62518436 -0.4183688 -0.7781448 0.7396012 0.6724459 0.21563418 -0.554694 0.56832445 -0.714445 -0.85129803 -0.19945419 0.33006954 0.81763965 -0.5041158 0.9011294 -0.48307687 0.61125755 0.44228718 -0.52010816 -0.8859032 0.7423768 0.5320138 0.52589417 -0.75778747 0.7601637 0.77443653 -0.7322521 0.65776443 0.34176108 0.4999639 -1.0157573 0.64378977 -0.90771455 -0.33697203 0.5127606 0.7941206 0.4283404 0.16617021 -0.45917064 -0.6769229 0.063261755 0.54286885 0.046924498 0.661635 0.36840296 -0.91034746 -0.6251082 -0.13069925 0.8235089 0.13825813 0.53713304 -0.21737581 -0.13280652 0.57631177 0.15705514 -0.55850214 -0.41695976 0.95724016 0.145987 0.38828182 0.8250053 -0.7581739 0.34250152 0.34700635 -0.2510866 -0.14841454; 0.9981224 -0.76867944 -0.65018994 -0.33541453 0.9145394 -0.7323732 -0.87925684 0.5994979 -0.5317116 -0.3671614 0.21436517 -0.5157152 0.7349244 -0.40930793 -0.47074488 0.5082098 -0.07011875 -0.31905958 -0.6710834 -0.7677628 0.3354939 -0.47039282 0.43664193 -0.022913247 0.534384 -0.9415474 -0.19245818 -0.6115332 -0.1515839 0.6939238 -0.6403146 -0.5870466 -0.66714716 0.3510422 -0.25468105 -0.6584858 0.07056583 -0.65288186 -0.4734592 -0.35635686 0.4999999 -0.18545641 -0.6193103 -0.27059278 0.26306248 -0.60555124 0.32175335 0.497935 -0.72928727 -0.11986235 -0.6772768 -1.0714513 0.7535414 0.44018498 -0.6811747 -0.2399465 0.67534 -0.044039544 -0.30796453 0.49887973 0.32507855 0.9831168 -0.6280088 -0.53890604 -0.26748854 0.79983276 0.72739345 0.44261572 -0.6900338 0.5417589 0.8328509 -0.5419264 -0.6582934 -0.5742233 -0.19714813 0.337709 -0.53287995 -0.84651875 0.935309 0.9902261; 0.9248805 -0.4718903 -0.6917191 -0.25645718 0.9239191 -0.7435906 -0.87316805 0.39661092 -0.65947926 -0.27711454 0.4385547 -0.9608361 0.7837159 -0.5902032 -0.25262773 0.30590534 -0.023549024 -0.5094788 -0.83053875 -0.63843024 0.57383037 -0.4527113 0.6264707 -0.6821163 0.3967327 -0.52715147 0.16194892 -0.8320681 0.07733054 0.456421 -1.0240558 -0.04576933 -0.56012124 0.67270595 0.012847977 -1.0016947 0.035411797 -0.5642505 -0.4970411 -0.03472763 0.52404326 -0.16601025 -0.7060424 -0.0303212 0.50307846 -1.1136825 0.5254187 0.77031547 -1.1088327 -0.23385638 -0.5386708 -1.2667359 0.95357794 -0.021823395 -0.6920411 -0.032086704 0.5282077 -0.09410105 -0.19717778 0.35780612 -0.012659676 1.136582 -0.84785044 -0.65170634 -0.12965167 0.85027677 0.40541825 0.29342237 -0.48421696 0.7269295 0.25510523 -0.33923206 -0.3052528 -0.7700635 0.12404402 0.35054225 -0.43921107 -0.74721396 1.2714635 1.0024196; 0.45230785 -0.42718616 0.21982966 -0.9498993 0.48488155 -0.36171925 -0.23915662 0.1779861 -0.19358107 0.9377061 0.62296623 -0.36235562 -0.050367486 -0.99210984 -0.48457584 0.17350712 0.5884926 -0.9437799 -0.61586094 -0.3857282 0.62969685 -0.85484445 0.78468853 0.87237644 0.25370744 -0.024768436 -0.71771735 0.425051 -0.8907958 0.36603367 -0.59262735 -0.49588135 0.576631 0.95855594 -0.58237654 -0.50417686 -0.8973658 0.830338 -0.7597477 -0.9419512 0.8805171 -0.5152568 -0.60180724 -0.58174497 1.028713 -0.35795793 0.93218845 0.5569375 -0.5814294 -0.6404169 -0.11696328 0.46180746 0.48236796 0.5414789 -0.051766362 -0.62485963 0.05024395 -0.7569143 -0.4292149 0.68777585 0.84628135 -0.18721212 -0.6192289 -0.024122609 -0.5343312 0.17527042 0.03939508 -0.6883607 -0.08331447 0.3863824 0.24347246 -0.91864634 -0.2876019 -0.30203086 -0.76784074 1.002622 -0.61766464 -0.13408832 -0.20993043 -0.28484747; 0.44387287 -0.65989286 -0.84727335 -0.16082028 0.42002952 -0.42721733 -0.41093454 0.80342036 -1.0291103 -0.49677932 -0.043264594 -0.28305563 0.56255835 -0.22161773 -0.34170488 1.1516166 0.10561531 0.06357187 -0.72640586 -0.5872904 0.04642362 0.042941716 -0.34562245 -0.25487858 0.95380896 -0.75382006 -0.17306428 -0.40172896 0.4794469 1.1918962 -0.3783104 -0.76298124 0.0013512973 -0.42975017 -0.3922654 -0.6201101 -0.36180484 -0.55835557 -0.013469019 -0.23764637 0.15824267 -0.65030944 -0.7107077 -0.5163625 -0.20147349 -0.15823165 -0.47808227 0.61413974 -0.68923724 0.2593183 -1.027402 0.09413647 0.410507 0.49274966 -0.7445564 0.08366125 1.2254156 -0.44082046 -0.97782516 0.41294217 0.38624164 0.3472846 -0.17864314 -1.0305405 -0.26286304 1.1067215 1.0905938 1.1221329 -1.0634446 0.8915396 1.1629587 0.78019327 -1.2890239 -0.7233375 -0.096686706 -0.17485656 -0.09879431 -0.648691 0.3316238 0.6179004; -0.39777142 0.20877406 -0.3183994 0.9775999 -0.38662732 0.31625423 0.57877856 -0.29954526 -2.1359883f-5 -0.96038306 -0.68154484 0.7704174 0.25527772 1.0211886 0.70081437 -0.15869181 -0.6533255 0.91108686 0.46835896 0.30488503 -0.5716812 0.9642485 -1.0150281 -0.7970419 -0.2799395 0.23306961 0.7452466 -0.48366463 0.74264675 -0.20796467 0.6381186 0.51270497 -0.4283584 -0.989985 0.48785597 0.40593547 0.71546155 -0.5153256 0.9524042 0.6631759 -0.95495784 0.6557261 0.4210018 0.49701187 -1.067474 0.6158227 -1.169126 -0.3761457 0.66697943 0.9826072 0.0014791595 -0.021598408 -0.6348465 -0.61149126 -0.21378201 0.6145491 -0.038862765 0.4772464 0.40462807 -0.41882408 -0.69999593 -0.14868522 0.70334864 0.044799592 0.5036467 -0.20778985 -0.097765766 0.81914914 0.022830883 -0.35034603 -0.28492072 1.0206517 0.27451056 0.24137089 0.72441256 -0.8867329 0.83576715 0.43902233 0.095302396 0.06668288; 0.8826377 -0.9189804 -0.6574981 0.37935877 0.8373687 -0.77148867 -0.8843006 0.4945597 -0.7156116 -0.50332385 -0.23571925 -0.5526842 0.6296643 -0.025265723 -0.3893833 0.83830625 -0.65327454 -0.08287917 -0.9238593 -0.6784604 0.05911188 0.48202336 -0.36881113 -0.4419879 0.7746728 -0.87954193 0.3832693 -0.49659455 0.41924432 0.71745807 -0.6475498 -0.35297272 -0.7348712 -0.42623073 0.08368031 -0.90340656 0.64231646 -0.58520395 -0.0014316887 0.46468973 -0.09399256 0.29212984 -0.66232574 0.22424641 -0.4724144 -0.956435 -0.55786896 0.8501201 -0.82149035 0.54331774 -0.89953315 -0.8040009 0.6729308 0.14788757 -0.76274854 0.5628788 0.6194545 0.37490886 -0.60898983 -0.05012444 -0.2723611 0.7218591 -0.12367064 -0.78267056 0.36674547 0.6824372 0.54668725 0.6691847 -0.57726616 0.7054932 0.807975 0.35728377 -0.87075543 -0.78548443 0.6131392 -0.28590783 -0.23402873 -0.7952288 0.7902912 0.6511507; 0.6045693 -0.8532637 -0.684602 -0.27133495 0.6539886 -0.6423051 -0.632015 0.64259624 -0.95554686 -0.59364116 0.28165647 -0.68653 0.8848375 -0.303756 -0.640005 0.87707746 0.21608382 -0.46000683 -0.8462104 -0.6429376 0.3594296 -0.33149952 0.14425117 -0.27034545 0.75947237 -0.87643987 -0.30646977 -0.69126 -0.13482249 0.72106415 -0.7894125 -0.444336 -0.1968717 0.21962748 -0.3945864 -0.73497057 -0.025116095 -0.5542455 -0.2751295 -0.4925454 0.3654891 -0.24237148 -0.827389 -0.35096303 0.085027575 -0.5375584 0.20534071 0.66049343 -0.6838947 -0.0029730774 -0.9269338 -0.53788024 0.56507397 0.66317123 -1.0317388 0.037664454 0.7372785 -0.4573696 -0.46926767 0.415984 0.23997751 0.74221617 -0.45144176 -0.77402896 -0.5482568 0.97458583 0.83963114 0.7880325 -0.96773833 0.86388105 0.6725569 -0.19569083 -0.819987 -0.6800711 0.023009561 0.009289944 -0.37829244 -0.9166534 0.5947885 0.9282932; -0.69783837 0.5718668 0.56461847 0.26612133 -0.8663394 0.42331707 0.7607874 -0.62921417 0.6311652 -0.54901814 -0.5504598 0.51399255 -0.8558336 0.51450104 0.48427933 -0.3458905 -0.48825744 0.29587978 0.6207086 0.769612 -0.38325202 0.41963714 -0.62003374 -0.64716756 -0.62902564 0.76211286 0.51993597 0.23874322 0.2635994 -0.5523936 0.64792496 0.46916822 -0.42524302 -0.5869864 0.5653019 0.4848951 0.6495175 0.90723103 0.37637314 0.58664304 -0.46154785 0.3641009 0.63698906 0.28724095 -0.35353172 0.68950987 -0.46802884 -0.42498848 0.85185385 0.47712228 0.46772864 0.0051771402 -0.80770797 -0.42305523 0.62161726 0.2733234 -0.9128872 0.690673 0.4477432 -0.5704926 -0.50398666 -0.6607143 0.4226718 0.6370297 0.46700108 -0.6073332 -0.73882425 -0.9935633 0.8282875 -0.7454732 -0.7734734 0.41789877 0.72665894 0.62387234 0.5909463 -0.29815912 0.5236272 0.7947008 -0.6804389 -0.7040715; -0.37251875 0.27482185 0.42745328 -0.89333147 0.07636579 -0.2355684 0.26900494 0.3712425 0.5065738 0.78764904 0.8228664 -0.76016897 -0.6190888 -1.0149109 -0.83283514 0.09088335 0.4113768 -1.0201166 -0.13994585 -0.17394336 1.0319668 -0.8096334 1.2024735 0.3588265 -0.07694565 0.54159313 -0.65385485 0.45398077 -0.69703096 -0.6397255 -0.51953185 -0.4503048 0.26004946 0.9490042 -0.5209397 -0.030889207 -0.33421662 0.5328862 -0.9929361 -0.74149275 0.90639806 -0.46486196 0.036861096 -0.68892 1.1694338 -0.22886027 0.9241985 0.39731926 0.1866668 -0.9016683 0.76184016 0.33799794 -0.10942034 0.69097966 0.609898 -0.831069 -0.91560346 -0.47722152 -0.24757259 0.7435614 0.5937147 -0.222561 -0.99358666 0.3776551 -0.5994179 -0.7059466 -0.73246455 -0.85247016 0.9003784 -0.2067379 -0.5657172 -0.85607284 0.5639425 0.13451105 -0.63687813 0.5378389 -0.69183034 0.5265195 -0.23012365 -0.34024492; -0.16713513 -0.23371372 0.36578697 -0.9209899 0.18028478 -0.6009502 0.13192143 0.25778213 -0.07906492 0.8247355 0.79784006 -0.5384877 -0.3064571 -0.67383415 -0.801775 0.36837333 0.74933267 -0.8310566 -0.04733008 -0.47596985 0.66167045 -0.8986522 0.5995676 0.824499 0.4249371 -0.11748225 -0.9024466 0.78239334 -0.90386385 0.20175405 -0.14679262 -0.75191283 0.6023694 0.7018796 -0.96090645 -0.08349578 -0.66819865 0.706244 -0.47440282 -0.85447717 0.5573987 -1.0099231 -0.39041632 -0.7804854 0.884233 -0.106465966 0.9528102 0.48611903 -0.0011723579 -0.7719839 -0.13712573 0.2008908 0.12822686 0.64444005 0.5802789 -0.6839022 -0.19260411 -0.83340377 -0.61731607 1.0722679 1.0254507 -0.68486327 -0.65538883 -0.30104047 -0.8337805 0.017865743 0.25450605 -0.61668944 0.3393289 0.24894895 0.37804875 -0.6381238 -0.20233329 0.028825311 -0.94149584 0.874538 -0.80085874 0.117173284 -0.61196065 -0.48946166; 0.4587011 -0.3956807 0.20335023 -1.0364879 0.625026 -0.6387571 -0.55480546 0.434349 0.01573342 0.9679743 1.0219903 -0.8695919 -0.429428 -1.104774 -0.43699536 -0.07181332 0.5890457 -1.0560256 -0.4540868 -0.46861953 0.82864314 -0.9706559 1.1044126 0.6673653 0.187183 0.042384367 -0.56239736 0.489783 -0.87920904 0.020624118 -0.54911554 -0.30551782 0.32830855 0.89976555 -0.53151464 -0.58857775 -0.43262085 0.62503135 -0.9744303 -0.8248863 1.0655675 -0.5435966 -0.3036085 -0.66360265 0.93424857 -0.7629048 1.1827133 0.35482523 -0.5058718 -1.0548111 0.2418238 -0.12387033 0.28318802 0.3983079 0.3161764 -0.7152222 -0.26361957 -0.23607554 -0.17214157 0.67045015 0.4929289 0.039961673 -0.9744749 0.036481768 -0.43196866 0.11316487 -0.18323444 -0.97690123 0.35119236 0.26740098 0.1621699 -0.98730874 0.05277146 -0.30284268 -0.7751766 0.86518323 -0.62517655 -0.1513488 0.118936054 0.011426545; -0.93035364 0.8720072 0.7841623 -0.044541437 -0.8494965 0.7563386 0.76208115 -0.55478734 0.8261593 0.5521115 0.003048553 0.76718116 -0.5844044 0.24288194 0.41405264 -0.6928686 0.19637159 0.16718192 0.5417994 0.7806435 -0.119413204 0.2234933 0.006124905 0.1273659 -0.7817327 0.87513584 -0.059602343 0.37282285 -0.22950494 -0.7992981 0.5029431 0.38496283 0.61026925 0.0942775 0.36394116 0.96265405 0.07205529 0.6958833 0.4296808 0.35724658 -0.21435724 0.31712037 0.9428364 0.45705992 -0.02016369 0.5524192 0.13050836 -0.75365406 0.8531987 0.034283675 0.94392866 0.39003655 -0.6080269 -0.38748068 0.7794298 -0.36617607 -1.0723727 0.05196627 0.5990686 -0.23735671 -0.37505138 -0.6993447 0.31724852 0.7333479 0.31951737 -0.7979116 -1.0521743 -0.57314616 1.0627215 -0.8079734 -0.67734957 -0.11214947 0.79476774 0.9218409 -0.22455367 0.19004215 0.55544615 0.77309185 -0.73855966 -0.5237149; -0.5535956 0.5719427 0.64795506 0.42951444 -0.568561 0.85331494 0.7494228 -0.3503485 0.5930868 -0.005849446 -0.34618428 0.7226846 -0.7385095 0.6485265 0.39157805 -0.77402854 -0.3626055 0.57398504 0.83328736 0.6851055 -0.5436248 0.5178588 -0.5343023 -0.12279658 -0.4420211 0.8403196 0.15749271 0.9969491 0.4361228 -0.65268 0.6988092 0.21225028 0.1433793 -0.57611924 0.21454363 0.7788365 0.47482622 0.8566051 0.4946776 0.3155835 -0.49865165 0.3793001 0.6957111 0.38679332 -0.53547436 0.8726655 -0.43606526 -0.5700818 0.74013525 0.3743107 0.6892744 0.74486256 -0.9059003 -0.5131103 0.8113051 0.28503928 -0.5708214 0.21344517 0.2585921 -0.35368195 -0.23403537 -0.89326614 0.38581052 0.49043205 0.30649155 -0.78104246 -0.5525593 -0.11555573 0.7926614 -0.53205335 -0.6069095 0.5356182 0.47627282 0.7492377 0.20375465 -0.3716227 0.60972804 0.61299616 -0.73330045 -0.81253684; 0.63408184 -0.6690066 -0.7295031 -0.092316695 0.6091493 -0.8375545 -0.8454177 0.62014526 -0.7131005 -0.3580297 0.054839406 -0.7283524 0.73807573 -0.45860064 -0.5444331 0.6311216 -0.30510446 -0.46731088 -0.5878297 -0.66775054 0.41251767 0.07959834 -0.0066926596 -0.20836137 0.6712237 -0.8815343 0.18145932 -0.38658115 0.44294363 0.8933881 -0.5375572 -0.39506376 -0.50245935 -0.13138556 -0.37226236 -0.99197006 0.024672104 -0.46515903 -0.09444017 -0.2348029 0.21915813 -0.32199287 -0.6693339 -0.448868 -0.035937518 -0.74060076 -0.1309851 0.7185909 -0.67598337 0.08233371 -0.940812 -0.4464469 0.9792893 0.58563113 -0.6952567 0.33189335 0.96738005 -0.12505536 -0.5134392 0.42460614 0.27066407 0.8378578 -0.43660977 -1.0033442 -0.3727855 0.8416761 0.97586375 0.58210546 -0.85332364 0.86956716 0.9474171 0.26051122 -0.9558554 -0.74176836 0.11188578 -0.17633662 -0.31848416 -0.6640587 0.5984669 0.5424062; -0.8498567 0.7237981 0.7501423 -0.5406357 -0.76793736 0.8455202 0.7223314 -0.7557512 0.67946535 0.59640956 0.3226232 0.84959465 -0.6351178 -0.04446637 0.3061979 -0.92720604 0.6071443 -0.043071497 0.79278195 0.68621755 0.03235451 -0.3176067 0.35835338 0.5029611 -0.77215356 0.86230737 -0.46042982 0.59092534 -0.64629626 -0.8236665 0.776521 0.22828257 0.7022119 0.46810812 -0.22315356 0.6587933 -0.5174063 0.45716777 0.12995675 -0.09395963 0.010848132 -0.22339898 0.9295958 0.018969169 0.50701874 0.68410856 0.6178918 -0.6321834 0.8032986 -0.60097003 0.7214455 0.5543659 -0.7364777 -0.24810028 0.69582474 -0.3952633 -0.7626584 -0.14043212 0.805313 0.0028850252 0.16440065 -0.50138664 0.10783545 0.96570635 -0.3595611 -0.81974477 -0.63224524 -0.7822674 0.83344454 -0.80893236 -0.68975616 -0.5497234 0.749872 0.94752514 -0.5591269 0.33368394 0.34087315 0.653564 -0.6121177 -0.54997855; -0.7916203 0.8057088 0.65883654 -0.4368273 -0.81349236 0.9244392 0.9468356 -0.5336921 0.714929 0.68393815 0.32341024 0.7227551 -0.8122055 0.16802858 0.541852 -0.8112597 0.42935148 0.15001228 0.6706674 0.9171156 0.045495003 -0.08943434 0.3260127 0.29096594 -0.92474693 0.646044 -0.35372278 0.48683676 -0.50944173 -0.76458675 0.5892738 0.4772155 0.68435776 0.103712566 0.031025 0.69078606 -0.35322553 0.57334334 0.014302205 -0.17982914 -0.13081798 -0.14307103 0.6975505 -0.0030994047 0.36488286 0.61250806 0.4433714 -0.82081175 0.92524904 -0.39186206 0.95843303 0.3347061 -0.71647054 -0.45930657 0.8436115 -0.3457905 -0.8043907 -0.2896083 0.5521122 -0.29759338 0.08182698 -0.4999254 0.08042024 0.85940397 -0.0627017 -0.97539693 -0.9946142 -0.631238 0.716815 -0.89876074 -1.001309 -0.34219927 0.7399264 0.76418036 -0.2833671 0.55649227 0.36770254 0.63608253 -0.56276363 -0.49323368; -0.1653482 -0.18766204 0.3677871 -0.6712172 0.16656171 -0.31623983 -0.25481716 0.27149424 -0.45697597 0.7693457 0.5678359 -0.37298018 -0.43489572 -0.49171758 -0.55436766 0.7032711 0.74426496 -0.52084047 -0.120923296 -0.4524019 0.541002 -0.7996752 0.54649585 1.0842516 0.5310928 -0.3033522 -1.0625712 0.6670234 -0.7364099 0.15073861 -0.26367554 -0.92358476 1.0214211 0.73578197 -0.90342385 -0.14516696 -0.78176355 0.6412758 -0.7308816 -0.8745549 0.68726814 -0.9264331 -0.105531916 -0.6680762 0.6199625 -0.12734593 0.775123 0.28188372 -0.27268428 -0.891098 -0.09870159 0.4425872 0.056238517 0.7917023 0.15406488 -0.83758575 -0.00899286 -0.89710814 -0.6016962 0.50201565 0.91774076 -0.38447174 -0.57607394 -0.14599687 -0.7672578 0.13294011 0.19532856 -0.3172688 -0.14356862 0.40625006 0.4865273 -0.7847208 -0.461763 -0.22214818 -0.75979424 0.82279533 -0.59421027 -0.25599146 -0.44609737 -0.42949945; -0.25327036 0.45937598 -0.42435715 0.8117981 -0.2944847 0.31170553 0.30598265 -0.36864093 0.22292422 -0.7706181 -0.86435837 0.40381107 0.39931908 0.55622643 0.53280026 -0.43633208 -0.9378563 0.70783687 0.22858317 0.22850093 -0.7185222 0.7600422 -0.66193944 -0.84322447 -0.63473237 0.10745966 0.9943466 -0.48727217 0.93772334 -0.43623942 0.24011756 0.7463329 -0.9762313 -0.54405403 0.65364647 0.29937255 1.0668904 -0.41380888 0.59429806 0.7923856 -0.7081677 1.029216 0.28594363 0.6279333 -0.7355306 0.18829595 -0.840707 -0.55601096 0.17314327 0.6579251 0.4266162 -0.69847935 -0.21296652 -0.5802462 -0.0788951 0.89544374 -0.047203213 0.9202904 0.37988758 -0.71887374 -0.8265256 0.29010317 0.5598217 0.49445337 0.7366925 -0.08556054 -0.19457002 0.5424239 -0.0011401417 -0.2090984 -0.25219864 0.617418 0.26686642 0.17461167 0.9423878 -0.8366771 0.81913537 0.06125708 0.29157555 0.099966735; -0.52006745 -0.01804962 0.47806188 -0.85697925 -0.31917876 -0.4890783 0.44712046 0.37970576 0.249363 0.71877223 0.577616 -0.4829221 -0.643345 -1.0276456 -0.54733795 0.08306488 0.7236619 -0.78567976 0.09916629 -0.23299453 0.72839373 -0.85212684 0.5664756 0.6081768 0.47737464 0.34443387 -0.7041766 0.5608029 -0.6711473 0.04245075 -0.116123915 -0.9747361 0.6796315 0.688023 -0.915998 0.4046272 -0.7907463 0.5983734 -0.91165024 -0.8740828 0.87173104 -0.8990654 -0.0506324 -1.0114907 0.67597336 0.22398947 0.57782835 0.32902253 0.25260177 -0.7449817 0.43441957 0.3650347 -0.487005 0.72377324 0.49336943 -0.5931772 -0.29370415 -0.7353024 -0.8328088 0.9298873 1.0698272 -0.6438599 -0.5586779 0.3612902 -1.0622203 -0.22419567 -0.23597486 -0.33531892 0.44188115 -0.18596725 0.19385989 -0.54638135 0.13968654 0.078054234 -0.65719736 0.7784281 -0.72092307 0.39646566 -0.42919412 -0.7376279; 0.6703901 -0.7874133 -0.5446795 0.14022514 0.85894954 -0.55692285 -0.59593534 0.7720462 -1.0311841 -0.6456795 0.009319148 -0.71535176 0.6295795 -0.3978449 -0.30648044 0.8132302 -0.25171077 -0.14410166 -0.7064123 -0.6503532 0.36509845 0.025866661 0.12856926 -0.28658015 0.9168912 -0.89653385 0.1311463 -0.49704763 0.46814752 1.0020436 -0.5445762 -0.46941656 -0.40456852 -0.038966566 -0.46990588 -0.8748204 0.36990294 -0.59313923 -0.2633885 -0.19399677 0.16615812 -0.27543947 -0.7432805 -0.1942438 0.08729498 -0.7895872 -0.15623236 0.610697 -0.66273785 0.081027456 -0.7620956 -0.523815 0.6578168 0.36160672 -0.7260823 0.42142293 0.91515964 -0.115648545 -0.34746063 0.46697134 0.38364327 0.7298431 -0.19000621 -0.9980741 -0.24420203 1.0103041 0.7900813 0.75285643 -0.9233129 0.8805447 0.8469953 0.30379808 -0.8584536 -0.9113801 0.1468795 -0.35913792 -0.2881334 -0.91454536 0.74847144 0.7129241; 0.8379013 -0.7043565 -0.48270115 0.37439242 0.8819746 -0.7357324 -0.963436 0.50621545 -0.36596328 -0.6011591 -0.28340212 -1.0211157 0.57958806 -0.01008781 -0.30558935 0.60220003 -0.667146 -0.21827665 -1.0124787 -0.7922778 -0.22446181 0.2815288 -0.34168005 -0.8033376 0.4125177 -0.56581855 0.6567334 -0.7141477 0.5386943 0.505605 -1.239858 0.3882616 -0.79176104 -0.24997139 0.42298523 -0.97994614 0.6114216 -0.65473175 0.1667571 0.61190856 -0.07282091 0.5669628 -0.83921087 0.59939873 -0.18884625 -0.81731254 -0.2272333 0.7825049 -0.9835437 0.35040802 -0.2629874 -0.80962574 0.9877612 -0.34467864 -0.5061748 0.5938617 0.3489108 0.5381056 -0.3108805 -0.6172531 -0.6375263 0.85736036 -0.3187565 -0.6724677 0.7974497 0.4397255 0.27012873 0.25172058 -0.30797353 0.6903348 0.44360337 0.23393 -0.59887964 -0.7596913 0.6052575 -0.3205389 -0.48101786 -0.6109462 0.68232685 0.9543093; 0.8589567 -0.5768859 -0.91446465 -0.27518997 0.66418874 -0.7746203 -0.80799055 0.56902385 -0.52976936 0.46748754 0.6322779 -0.5366751 0.9737007 -0.5044743 -0.50737554 0.6139317 0.38714153 -0.5956241 -0.6604762 -0.5653766 0.3624363 -0.47286847 0.54304546 0.28551736 0.4744901 -0.8261844 -0.3381922 -0.74622554 -0.23005103 0.5695478 -0.84359807 -0.34708422 0.3821995 0.49653876 -0.3081472 -0.78497463 -0.40313926 -0.94566846 -0.29393968 -0.26096174 0.31506306 -0.4029269 -0.72982717 -0.34676635 0.5985805 -0.7542008 0.54297477 0.67409396 -0.5987095 -0.3634698 -0.5816638 -0.7994505 0.7261664 0.4473104 -0.8689031 -0.23366007 0.56389797 -0.3817094 -0.2931548 0.5872693 0.43921077 0.6957407 -0.51410866 -0.7362195 -0.3696415 0.7721511 0.5787865 0.35009345 -0.64457077 0.6444011 0.45745528 -0.4493003 -0.50166124 -0.80707836 -0.30006734 0.27658156 -0.5203583 -0.8223461 0.79723257 0.81817377; 0.53927416 -1.047153 -0.8363604 0.31144506 0.61658895 -1.0343868 -0.7016671 0.61442596 -0.98438805 -0.5503643 -0.11564921 -0.5423529 0.87054855 -0.1953309 -0.46834913 0.7428508 -0.53626275 -0.15514137 -0.90522605 -0.91301733 0.041152135 0.41874918 -0.31195417 -0.33776522 0.79339725 -0.9460245 0.3132029 -0.66478664 0.4260479 1.06608 -0.74947274 -0.17329255 -0.6828764 -0.17831129 -0.16837081 -0.54735893 0.24870604 -0.67158717 -0.033057492 0.009747307 0.15242842 0.08506097 -0.87631994 0.16140641 -0.20272958 -0.73510194 -0.5342074 0.6517597 -0.5982836 0.5620752 -0.986302 -0.56554484 0.88410205 0.3955338 -0.6845544 0.5019174 0.93958294 0.3856698 -0.47215027 0.4067642 -0.002719875 0.6052144 -0.1319879 -1.0033029 -0.014683413 0.8667162 0.8364911 0.8964868 -0.58758307 0.69568384 1.0930965 0.39581233 -1.0891719 -0.8949753 0.38720062 -0.42912704 -0.36228403 -0.86760217 0.7358323 0.5009875], bias = Float32[0.20846823; -0.5329952; 0.63974017; 0.91367346; -0.8397736; -0.09144847; -0.53783685; 0.78549665; -0.2836297; -1.0226449; 0.74578464; -0.54131037; -0.25151554; 0.33537734; 0.44486624; -0.29544696; 0.026663762; -0.2067949; 0.45008978; -0.36772412; -0.17294726; 0.5029558; 0.596315; 0.16724864; -0.9014451; 0.72556686; 0.2620261; 0.97060865; 0.7436266; 0.6959137; -0.8129347; -0.9022859; -0.59363437; -0.3100282; 0.20630597; 0.38628003; -0.41159227; 0.633024; -0.43994072; 0.37258357; -0.2721119; 1.0796156; -1.0310245; 0.15516536; -0.53970766; 0.21702118; -0.55503297; 0.3351382; 0.92267096; -0.6542948; -0.2433379; 0.5085052; -0.81510836; -0.9864596; -0.7325643; -0.8017186; 0.7699931; -0.3827987; -0.6440246; -0.8125297; 0.14032078; 0.9197489; 0.25265673; -0.2631483; 0.47964457; -1.0787419; -0.66888195; -1.0625547; 0.089244835; 0.5079339; -0.03882152; -0.33019817; -0.22524057; -0.6875611; 0.7321587; -0.6692797; -0.0004045116; 0.17996837; -0.4622005; 0.38578582]), layer_3 = (weight = Float32[0.10822653 0.75300306 0.44534215 -1.0660635 1.029667 -0.404766 0.16688171 -0.3198062 -0.3711247 -0.07393741 -1.078655 0.7456551 0.8586201 0.54062164 -1.0563141 -0.34855068 -0.74146956 -0.76042724 0.02027175 0.6289235 -0.7487474 0.32872695 -0.2186801 -0.21483219 1.0126138 0.5000794 -0.16728564 -1.07841 0.791773 -0.52122325 -0.6196445 0.8353792 0.32170644 -0.50751454 -0.3004143 0.37549943 -0.030568995 -0.8048329 0.0064890836 -0.7631586 0.09434857 -0.55368876 0.6958149 0.6359698 -0.5526362 0.030728068 1.2441335 -0.9896565 -1.0871336 0.6944893 0.99364597 -0.013104238 -0.9587082 0.29740298 0.9586639 -0.82266587 -1.2807791 -0.14723766 -0.11223302 1.0140953 -0.7352876 -1.0495775 -0.9869243 -0.6411948 0.068230525 1.2060789 0.97549576 1.3053463 0.70435226 0.4956665 0.16530418 1.0379565 0.6180946 -0.8750655 0.4555296 0.32034367 -0.34445402 0.057547063 -0.47250378 0.011408482; 0.065939024 -0.22478467 0.0056512766 -0.24780045 -0.9821209 -0.2119601 0.94172454 -0.59998494 -0.57306904 0.14491385 1.0788872 -1.0697796 1.1858302 0.9841079 -0.13713272 0.5130602 -0.84032595 -0.1678359 0.0676785 0.024776056 -0.46823323 0.45423234 1.2508053 1.3527437 -0.9174003 0.138734 1.6182247 -0.71287334 0.23842701 0.7596515 -0.045104742 -0.80379874 -1.755412 -1.3605133 0.124377884 1.1749363 -0.054285523 0.5759056 0.9048268 -0.037391003 -1.7370765 -0.6386058 -0.7714344 0.39330894 -0.47951594 -1.284515 -0.32321796 -0.25884447 0.5248736 -0.88607705 0.2589883 0.7947201 0.8690946 1.2032312 -0.64723015 -0.32048342 0.92075485 1.1228977 0.5744841 -1.0904721 -0.26909485 0.51067525 -0.057240915 -0.5730655 -0.2749184 0.3806325 -0.65736145 -0.43392763 0.46974283 -0.5544087 0.031839773 0.36901066 0.124544404 -0.3623773 -0.081302255 -0.7378889 -0.14630057 0.81494325 0.44957355 -1.5164119], bias = Float32[-0.98429525; -0.26745918]))
loss_neuralode(p)


#----------------------moderate noise loss collection 

rng = Random.default_rng()
Random.seed!(99)


x1=Array(sol)

x1_mean = mean(x1, dims = 2)
noise_magnitude = 7e-2
x1_noise = x1 .+ (noise_magnitude*x1) .* randn(eltype(x1), size(x1))
#Displaying true data vs noisy data



#-------------------------------------Defining the Neural ODE------------------------------------


dudt2 = Lux.Chain(Lux.Dense(2, 80, tanh),Lux.Dense(80, 80, tanh), Lux.Dense(80, 2))
#Setting up the NN parameters randomly using the rng instance
p, st = Lux.setup(rng, dudt2)



prob_neuralode = NeuralODE(dudt2, etaspan, Tsit5(); saveat = etasteps)


function predict_neuralode(p)
    Array(prob_neuralode(I, p, st)[1])
end


#training data
true_data= x1_noise[:,1:end-20]
### Define loss function as the difference between actual ground truth data and Neural ODE prediction
function loss_neuralode(p)
    pred = predict_neuralode(p)
    loss = sum(abs2, true_data .- pred)
    return loss, pred
end
p_trained_moderate = (layer_1 = (weight = Float32[-0.57100105 0.015194183; -0.54241246 -0.83212835; -0.39792842 -0.30181402; 0.45323196 0.4437404; 0.45289892 0.4539182; 0.76053184 0.55429167; 0.38485485 0.7572667; -0.3242625 -1.0920783; -0.6685054 -0.53784215; -0.91893 -1.1385946; 0.7541218 0.6168198; 0.5900667 0.9702902; -0.5424533 -0.7102526; -0.41596437 -0.63901436; 0.7369241 0.61125386; -0.7218074 -0.47311538; 0.29003736 0.36463612; -0.4554555 -0.3555462; -0.5454643 -0.91457486; -0.42628312 -0.6320439; 0.17450106 0.7913556; -0.5292868 -0.8021426; -0.8106068 0.43135414; -0.75656253 -0.3918693; -0.58533114 -0.8553635; 0.23427811 0.3338823; -0.46867105 -1.5353473; -0.7515417 0.04316143; -0.81317157 -0.8183972; 0.5274718 0.23065187; -0.67978555 -0.42957953; -0.7613024 -0.37252736; -0.5593862 -1.0940838; -0.37844428 -0.29890344; 0.45773932 0.4957841; -0.73720396 -0.40645775; -0.67518526 -0.5687117; -0.8006008 -0.32384625; 0.43219665 0.62776834; -0.58120114 -0.22970153; -0.40698868 -0.32141122; -0.3985983 -0.4702598; -0.645558 -0.83808064; -0.55553925 -0.5482841; 0.41766292 0.5713099; -0.5896965 -0.46804786; -0.39500374 -0.7764357; 0.5061121 0.63586074; -0.7699743 -0.377371; 0.5221207 0.5258538; 0.3786207 0.8316812; 0.47047415 0.61111385; -0.53987026 -0.49142078; -0.42370546 -0.63839877; -0.484164 -0.7811067; -0.7728576 -0.84289765; -0.61950964 -0.55811864; 0.46283022 0.88927466; 0.801724 0.33450508; 0.47211665 0.58693373; -0.31752074 -0.67859286; -0.4398437 -0.32657653; -0.6041642 -0.30561304; -0.69809765 -0.5937141; -0.052609216 -0.6599864; 0.7502642 0.4727341; -0.68818456 -0.642879; -0.6230211 -0.6032567; -0.64063954 -0.503592; 0.59165066 0.82086694; -0.40578195 -0.3877862; 0.6443767 0.7847991; 0.6459048 0.16194594; 0.4236042 0.6952767; -0.72394025 -0.49035874; -0.61381584 -0.5494507; 0.8467544 0.78093696; 0.42116344 0.39282835; 0.46574613 0.6553157; 0.65147656 0.82457703], bias = Float32[-0.6241085; -0.48318812; -0.5969034; 0.59943044; 0.5715209; 0.59562; 0.60430795; -0.3978772; -0.5030613; -0.22006784; 0.66245216; 0.48751125; -0.6482156; -0.62330526; 0.5602052; -0.58137286; 0.6718932; -0.59539837; -0.6045714; -0.6552925; -0.55314183; -0.62424225; -0.44492128; -0.61629283; -0.51123637; 0.7024261; -0.31258157; -0.52571297; -0.6361366; 0.40744045; -0.5784888; -0.5915465; -0.57916826; -0.58074373; 0.53829986; -0.61342967; -0.6734077; -0.62225175; 0.5992204; -0.48977387; -0.5437869; -0.5972875; -0.6390128; -0.5690812; 0.5795446; -0.55947906; -0.54290485; 0.6347318; -0.61268985; 0.57624876; 0.58117926; 0.6024125; -0.53865606; -0.6358035; -0.6619017; -0.6576789; -0.55567557; 0.6353807; -0.012483071; 0.5840312; -0.63379127; -0.6164488; -0.5220328; -0.4761625; -0.20296958; 0.52808696; -0.65666795; -0.6201634; -0.55982983; 0.630973; -0.55815583; 0.53467506; 0.51065105; 0.6525808; -0.5651441; -0.550624; 0.6555221; 0.54631716; 0.6369743; 0.6650506]), layer_2 = (weight = Float32[0.6012523 0.6434683 -0.5957349 0.46099555 0.55850166 -0.39826015 -0.764842 0.5057613 0.54897517 0.65462404 -0.61303943 -0.57506776 0.4942306 -0.53667516 -0.5147804 -0.46282986 0.48804158 -0.47595367 0.53961456 -0.59260905 -0.4102284 0.493714 0.5100217 -0.4660888 0.399498 0.54808617 0.4748284 -0.6875272 0.5761173 0.53741586 -0.46147388 -0.48353425 0.62857 -0.57238525 0.51454794 -0.5459514 0.5154212 -0.7235261 0.5220057 -0.736895 -0.65606934 -0.5613972 0.43700495 -0.41661575 0.4580715 -0.7470254 0.5453025 0.53405005 -0.5361883 -0.7243048 -0.78371334 0.7013627 -0.55598336 -0.6309172 0.70462745 0.6016913 -0.4474614 -0.682687 0.578189 0.40938976 -0.7049371 -0.6843186 -0.7814952 0.49576518 -0.7778033 -0.5838258 0.6258373 0.49683166 -0.5184375 -0.4145851 -0.63231665 -0.7595026 0.77773714 -0.7344617 0.59391505 -0.69558555 -0.70471776 0.5375308 0.58480954 -0.4973678; 0.70969003 0.48344278 -0.6669096 0.5406615 0.4336663 -0.43441096 0.4503477 0.59655994 0.4268625 0.66038173 -0.6798299 -0.6362337 0.65820813 -0.5277716 -0.50983584 0.6884472 0.7704499 -0.58634466 0.5701274 -0.7106144 -0.47402427 0.6988273 0.56873727 -0.5964597 0.43346354 0.5363263 0.60308576 -0.67115396 0.7275298 0.56124216 -0.60959405 -0.48084804 0.6209096 -0.5985139 0.5850407 -0.687596 0.71564394 -0.66084176 0.5121948 -0.7088621 -0.5121149 -0.4252161 0.41629666 -0.45726952 0.561672 -0.78064036 0.6631048 0.7164321 -0.5832624 -0.6812939 -0.7995217 0.5423392 -0.42804524 -0.53127354 0.5402328 0.6569756 -0.69416374 -0.75211644 0.39571095 0.6477748 -0.6903191 -0.6831466 -0.7586113 0.6771366 -0.5702461 -0.69378567 0.7428001 0.69411665 -0.4145911 -0.77987176 -0.6849345 -0.5805121 0.64625025 0.47772142 0.66369486 -0.6278309 -0.5023297 0.6440658 0.4500905 -0.70911115; -0.5812605 -0.5854772 0.62634104 -0.57689923 -0.52664864 0.20267962 0.80176556 -0.6276732 0.4022675 -0.45351368 0.71553516 0.63278437 -0.73174775 0.033467446 0.47075662 0.5161659 -0.64092326 0.408317 -0.6377073 -0.43388367 0.4203479 -0.68747705 0.6966135 0.63143754 -0.7219073 -0.47138238 -0.7419651 0.6325006 -0.7937477 -0.5124227 0.4020548 0.41756928 -0.49889112 0.4247223 -0.7434693 0.38755053 -0.70883554 0.40749875 -0.27053025 0.57895684 0.7390296 0.646748 -0.5513994 0.41896406 -0.49985006 0.42680097 -0.7624491 0.8030222 0.40749076 0.43781877 0.57191765 -0.28598455 0.6630813 -0.48673296 -0.8085996 -0.67532307 0.517668 0.67787826 -0.5320861 -0.52897406 -0.62887526 0.6886682 0.670963 0.3114111 -0.77308977 -0.55911654 -0.6945707 -0.62801147 0.4289207 0.5324531 0.6741115 0.43297377 -0.41787687 0.6807839 0.40947485 0.5791433 0.75128293 -0.65938795 0.7698238 0.80245924; 0.5668973 0.4272087 -0.39938053 0.729073 0.5682634 -0.36496097 -0.7173225 0.38673967 -0.74127036 0.52152544 -0.6236919 -0.5087331 0.5854045 -0.61275536 -0.54720515 -0.6085168 0.6942547 -0.57965636 0.5365939 -0.5336112 -0.63837355 0.6246583 0.23359454 -0.70312977 0.41343638 0.47637463 0.6787632 -0.4307403 0.5934296 0.73969495 -0.5126301 -0.72209954 0.6678159 -0.7779954 0.45183784 -0.71068215 0.51870847 -0.5755724 0.5677076 -0.6941636 -0.67062855 -0.5141553 0.4040849 -0.45635697 0.58496433 -0.45494255 0.6831648 -0.17279136 -0.5171647 -0.67021465 -0.42671937 0.5581432 -0.6098239 -0.79756266 0.60189056 0.65036553 -0.47991824 -0.5473697 0.5570206 0.6441834 -0.70464927 -0.42464212 -0.4704864 0.24722685 -1.4344095 0.679784 0.44536546 0.23831709 -0.5575005 -0.7257519 -0.5458342 -0.439886 0.48729587 -0.52874035 -0.6095875 -0.713537 -0.364796 0.49528953 -0.19035459 -0.6076262; 0.84087557 0.7791273 -0.6579402 0.44282874 0.47887942 -0.6153707 0.47988138 0.6682864 0.62133133 0.61996377 -0.5756904 -0.788824 0.8125252 -0.5721398 -0.68378663 0.5037334 0.56023735 -0.47249782 0.47865131 -0.6211092 -0.41404608 0.72471255 0.66095763 0.79699534 0.6445683 0.7077693 0.701825 -0.51072544 0.7762563 0.5512289 -0.6226426 0.6486304 0.6705169 -0.53737885 0.44386867 0.75529784 0.5746193 0.77904165 0.392731 -0.5304699 -0.58103675 -0.5134733 0.45984355 -0.6115072 0.6281634 -0.7001315 -0.36383608 0.6464254 0.68627465 -0.69555026 0.7480613 0.39810193 -0.4476921 -0.5114981 0.59517086 0.79653305 -0.40127525 -0.44380897 -0.44583642 0.509925 -0.72373515 -0.4341923 -0.59101343 0.7091712 -0.5406651 -0.6433858 0.78936386 0.44172728 -0.43111062 -0.7024937 -0.53242636 -0.59441626 0.66697544 0.73233443 0.678958 -0.52092564 -0.6861758 0.5605119 0.58523464 -0.43834168; -0.39756778 -0.64330995 0.43542838 -0.49147126 -0.72269666 0.69843024 0.74849784 -0.53864986 -0.04601921 -0.5096754 0.6508077 0.7004997 -0.5923137 0.53385174 0.6313894 0.72713584 -0.5462071 0.60241526 -0.38944304 0.41913736 0.76231307 -0.61918163 -0.68536025 0.7016662 -0.5729618 -0.76724344 -0.7318543 0.4259651 -0.39979723 -0.43162423 0.5296582 0.66119915 -0.5082447 0.72353095 -0.7792229 0.7641784 -0.72131807 0.45453364 -0.65650326 0.74852306 0.5945201 0.65532094 -0.6749231 0.522225 -0.53209853 0.44965714 -0.49709034 -0.71075094 0.718482 0.74052024 0.44984517 -0.76851875 0.5354467 0.6474734 -0.5498297 -0.4094558 0.46640697 0.6181688 -0.7978648 -0.47104478 0.44610786 0.5101994 0.5821696 -0.44081512 0.89095867 0.2946691 -0.41582426 -0.34057987 0.6131083 0.46005875 0.44910452 0.41157082 -0.49584305 0.25500768 -0.4218778 0.7146286 0.7323272 -0.67369634 -0.7182919 0.5992283; 0.62199336 0.46989194 -0.5651197 0.4353559 0.42906412 -0.7666073 -0.56824505 0.6206907 0.7762598 0.721436 -0.49368304 -0.7289478 0.48133656 -0.54597175 -0.6242231 -0.71806127 0.43557125 -0.62653077 0.65499735 -0.5562493 -0.51720935 0.78754616 0.6410636 -0.51625496 0.67184484 0.78926015 0.42474172 -0.47871646 0.681013 0.5068074 -0.6912295 -0.5970331 0.41959536 -0.66274226 0.44694966 -0.6160227 0.7755608 -0.55380756 0.5183635 -0.39373007 -0.397703 -0.6073741 0.6770252 -0.45257804 0.5826343 -0.6611125 0.52877873 0.42139626 -0.46661982 -0.63581663 -0.6212025 0.688803 -0.48808137 -0.72302014 0.52364355 0.45587695 -0.55325097 -0.40249914 0.53461385 0.63139254 -0.70588946 -0.40878606 -0.6635996 0.56022155 -0.8771411 -0.71824276 0.49073076 0.7524631 -0.45002997 -0.780403 -0.7687479 -0.47700146 0.60389465 -0.9959812 0.6873798 -0.42323744 -0.74698573 0.61784756 0.6320831 -0.7321245; 0.5803379 0.43311733 -0.6241611 0.61073536 0.5773302 -0.61858946 -0.77961123 0.6270589 0.50657004 0.4497063 -0.75321627 -0.7027805 0.53866535 -0.50451815 -0.6543748 -0.37068403 0.43246323 -0.42408934 0.48189214 -0.40701434 -0.41179165 0.54402757 0.7383555 -0.4044105 0.45822436 0.5668462 0.64651746 -0.63507044 0.41212165 0.54487306 -0.7294001 -0.44098952 0.7833155 -0.7629393 0.5924717 -0.43936127 0.8063173 -0.77294976 0.605376 -0.6288834 -0.42128125 -0.725929 0.61157084 -0.3865555 0.52878743 -0.39622173 0.48078406 0.3941435 -0.42774656 -0.75118 -0.73319536 0.439023 -0.4111631 -0.3904822 0.71450824 0.54222983 -0.2831561 -0.6747403 0.51137835 0.59245986 -0.75693667 -0.6719026 -0.7110324 0.5656718 -0.8703369 -0.3233816 0.44795677 0.60783774 -0.5259235 -0.49075103 -0.66295904 -0.39918765 0.443607 -0.65472263 0.8851474 -0.6451726 -0.52816355 0.4724983 0.59163994 -0.5631552; -0.6728458 -0.5888533 0.57039875 -0.5043117 -0.70392746 0.7368949 -0.6874795 -0.60115916 -0.64924055 -0.585518 0.6439554 0.5559678 -0.63992995 0.4335355 0.44370744 0.8035611 -0.54992753 0.57908386 -0.47222638 0.48875278 0.4590394 -0.57479405 -0.547131 0.6252431 -0.6782807 -0.56382257 -0.42019844 0.7809173 -0.46168837 -0.4914284 0.42920718 0.45049155 -0.7681132 0.4578696 -0.43380168 0.50843745 -0.47282586 0.7080023 -0.43786657 0.6918754 0.5427776 0.77308846 -0.57786274 0.63192236 -0.71410096 0.47510874 -0.40609688 -0.54253286 0.6693717 0.55658567 0.6704876 -0.5008659 0.6798028 0.50377744 -0.6288884 -0.7381512 0.47556245 0.5701602 -0.5691829 -0.6980284 0.49102423 0.59183156 0.5235213 -0.49880648 0.621854 0.66991717 -0.44377708 -0.39794576 0.650063 0.50752527 0.62138 0.7651479 -0.66674167 -0.71768636 -0.7035779 0.68823695 0.6282843 -0.77795583 -0.6519649 0.61067593; -0.5024439 -0.5443136 0.7424527 -0.6315118 -0.5847458 0.7166387 -0.7071523 -0.71887213 -0.39366212 -0.54480106 0.64581776 0.49660686 -0.5083809 0.54124033 0.39187708 -0.4102966 -0.75080365 0.5624826 -0.44291022 0.5616392 0.40894625 -0.6806849 -0.63129646 0.53285587 -0.52227473 -0.7559219 -0.39834988 0.6145241 -0.6602219 -0.7947031 0.75691354 0.4453445 -0.7344561 0.46923086 -0.4191475 0.6073471 -0.5421301 0.6022215 -0.56036294 0.50246674 0.7393589 0.587779 -0.49217507 0.536791 -0.63911206 0.68451244 -0.6701484 -0.5855162 0.64821655 0.48528373 0.39982843 -0.4230507 0.5198921 0.61697614 -0.4108754 -0.6641407 0.61999863 0.5253671 -0.48157886 -0.7728554 0.7423579 0.4852543 0.42835227 -0.5554935 0.7119671 0.58613133 -0.58942413 -0.53433985 0.67544633 0.735373 0.4829349 0.40127844 -0.5219035 -0.7105314 -0.5692116 0.56053954 0.7561064 -0.78109163 -0.70846045 0.48118514; 0.46385756 0.6105134 -0.48932558 0.656714 0.40989622 -0.6293185 -0.4267971 0.489348 0.77993023 0.44170237 -0.64438283 -0.42157388 0.7427993 -0.62028533 -0.63811946 -0.08969549 0.7238508 -0.46078825 0.4663274 -0.565295 -0.5798664 0.7412759 0.5848343 -0.40848735 0.6343973 0.46215978 0.41453248 -0.68513924 0.57848567 0.47394902 -0.41737738 -0.47819906 0.7747739 -0.49795476 0.5182062 -0.4447419 0.5762473 -0.5384466 0.5049986 -0.47507346 -0.6331948 -0.69993985 0.74910855 -0.5537915 0.48792785 -0.4183103 0.44296014 0.44088805 -0.50117886 -0.53520614 -0.68765146 0.5973269 -0.54162145 -0.74832845 0.76223844 0.6919637 -0.52543086 -0.6001915 0.60166013 0.52429724 -0.7053898 -0.4906875 -0.48172817 0.78935206 -0.6983446 -0.8096515 0.61393136 0.6680294 -0.58237964 -0.75486535 -0.5840595 -0.41761917 0.65089196 0.39737043 0.4836475 -0.5625853 -0.63067394 0.58667785 0.5871147 -0.41843176; 0.7163887 0.45717373 -0.7288296 0.56031036 0.522268 -0.7341337 -0.68783873 0.540245 0.54588467 0.4364714 -0.65971416 -0.54834765 0.5395121 -0.47710347 -0.7749994 -0.6427721 0.7135203 -0.49646643 0.6146762 -0.5625156 -0.6005106 0.5239448 0.65897155 -0.6484492 0.56144255 0.41992927 0.6988686 -0.7820492 0.5705529 0.6420736 -0.45448238 -0.54196954 0.5375868 -0.74840164 0.6283168 -0.51476675 0.55680984 -0.5507737 0.42594704 -0.6679344 -0.4939959 -0.6945572 0.5448214 -0.47801232 0.5415773 -0.4413159 0.5762031 0.74720055 -0.5250571 -0.6280612 -0.763351 0.4483623 -0.43400532 -0.69025564 0.7654647 0.56111866 -0.74269706 -0.59770435 0.6640306 0.41396502 -0.63106984 -0.5993865 -0.7837545 0.53193945 -0.49289265 -0.7620368 0.49580738 0.6958192 -0.42842382 -0.48484504 -0.4277054 -0.64838684 0.54669017 0.44421154 0.6837201 -0.7159402 -0.7531704 0.6512589 0.6703568 -0.5658449; -0.5067186 -0.40195692 0.6131394 -0.60203165 -0.47256473 0.5664744 -0.7143412 -0.6102723 -0.3773705 -0.72703695 0.47361565 0.51890403 -0.7590728 0.60687435 0.51145 -0.5502292 -0.69131356 0.51734096 -0.55290246 0.6179375 0.6616023 -0.5479968 -0.4474867 0.770094 -0.59009445 -0.43356094 -0.5840214 0.7751468 -0.54749215 -0.76488775 0.71863765 0.5578316 -0.43189138 0.6742796 -0.6754158 0.5527073 -0.6516137 0.6546514 -0.6406779 0.5011957 0.43952465 0.7555181 -0.4428376 0.6511451 -0.4213142 0.72811604 -0.46601406 -0.7459862 0.44909865 0.53062505 0.62951094 -0.6922898 0.44256255 0.4521456 -0.4710345 -0.7526168 0.75885326 0.67959255 -0.62856334 -0.63500106 0.7064305 0.5492267 0.4993075 -0.50088334 0.59565777 0.65929884 -0.5195698 -0.76992464 0.55950904 0.78242046 0.5947004 0.59548336 -0.4208993 -0.47508818 -0.6441593 0.49847004 0.7010749 -0.60954946 -0.46951726 0.54727215; -0.7780521 -0.7110909 0.47600546 -0.40168107 -0.71578467 0.79019547 -0.5899359 -0.54795635 -0.6613569 -0.4747462 0.6442319 0.73142976 -0.81587476 0.5942185 0.7150663 -0.03038398 -0.68176293 0.7453844 -0.73351663 0.48224625 0.6940376 -0.5012299 -0.7633787 0.5910413 -0.4950956 -0.7316887 -0.51067215 0.68889725 -0.70847327 -0.635054 0.61782503 0.70017356 -0.49244642 0.74429435 -0.5767867 0.73466164 -0.57197917 0.47637329 -0.76837796 0.6051691 0.6972861 0.67952704 -0.64008176 0.49536455 -0.5278461 0.40547705 -0.484914 -0.52685803 0.61344874 0.46707237 0.79966897 -0.6374431 0.42464736 0.6274409 -0.46892348 -0.6503175 0.5283717 0.6797478 -0.66956395 -0.7598625 0.6017529 0.6210979 0.69138753 -0.7651974 0.47857022 0.6440803 -0.6320168 -0.79255825 0.6461435 0.71490663 0.71747833 0.48281178 -0.52984077 -0.48735392 -0.77905935 0.58426094 0.7197429 -0.56619996 -0.5639129 0.43411854; 0.4938572 0.48979506 -0.6900409 0.5858592 0.49163136 -0.5324788 0.6899906 0.62035584 0.58420235 0.4194927 -0.6480853 -0.4527791 0.54350436 -0.5551928 -0.43140858 0.5605509 0.44658467 -0.48878282 0.6821882 -0.55962247 -0.56301004 0.75396276 0.7263663 -0.4523305 0.665386 0.76560354 0.54653776 -0.61522657 0.52299 0.73419875 -0.45321807 -0.60525006 0.60870177 -0.5687038 0.5993704 -0.57264423 0.6603049 -0.53909093 0.5495204 -0.78795224 -0.55750495 -0.7843384 0.6500748 -0.7345173 0.4976659 -0.51397616 0.5121258 0.41273773 -0.4916084 -0.53698176 -0.6767327 0.47060266 -0.7541428 -0.6546975 0.6828354 0.59455705 -0.49661502 -0.48937935 0.5395675 0.46596763 -0.631412 -0.50218475 -0.4528974 0.604506 -0.61223495 -0.42920932 0.48434436 0.69704676 -0.5023164 -0.5304585 -0.51062155 -0.4027115 0.4920262 0.58267915 0.6891302 -0.6003892 -0.7323764 0.4255647 0.6941853 -0.58873296; -0.64300495 -0.40006372 0.50666624 -0.4612079 -0.7040339 0.6102517 0.78459346 -0.5066428 0.6053165 -0.76021177 0.4905524 0.7316156 -0.7287472 0.43020236 0.7378743 0.57067335 -0.75663894 0.4013649 -0.41503727 0.58580774 0.5894663 -0.76867306 -0.38704586 0.410038 -0.70954734 -0.4304406 -0.57396424 0.43203425 -0.61581117 -0.53083956 0.75225806 0.53631085 -0.5854637 0.73376226 -0.6937406 0.7185807 -0.6450377 0.60192657 -0.39753252 0.59513927 0.46548495 0.6104393 -0.5734924 0.6995748 -0.6328622 0.70520544 -0.4900251 -0.18265694 0.7647387 0.56521755 0.7625584 -0.65435225 0.7587403 0.7683195 -0.72400284 -0.66126674 0.78007764 0.59537435 -0.4647867 -0.61243236 0.49952444 0.6094283 0.44135928 -0.3955588 0.875365 -0.6178917 -0.5450812 -0.46095827 0.6065656 0.5281059 0.7528905 0.62136996 -0.69216484 0.4664176 0.69266677 0.4983919 0.4256116 -0.64080864 -0.67330986 0.4153164; -0.44671345 -0.6945766 0.5102403 -0.5460171 -0.4263213 0.49469745 -0.41996604 -0.7823412 -0.3936303 -0.41941258 0.584921 0.7315119 -0.7831611 0.5851598 0.566914 0.05618161 -0.45138827 0.44629186 -0.41254094 0.6062858 0.41659337 -0.4485288 -0.6124824 0.4661532 -0.45432955 -0.5001304 -0.49699646 0.5285337 -0.42668554 -0.66719294 0.50646454 0.6921793 -0.7138797 0.39884385 -0.43295726 0.45145148 -0.4175238 0.7197814 -0.56488025 0.6462782 0.62132424 0.75989056 -0.690631 0.7814532 -0.4287545 0.59352493 -0.49835047 -0.5027079 0.43865627 0.54847765 0.64227104 -0.68230265 0.44341007 0.5065511 -0.6404084 -0.528684 0.74713135 0.473775 -0.6318385 -0.40350163 0.72361904 0.44785452 0.7818788 -0.691402 0.6259482 0.598931 -0.44167912 -0.6511343 0.51139 0.6519192 0.5419575 0.5897975 -0.68383753 -0.4540387 -0.7032129 0.74241436 0.5881485 -0.6152475 -0.4671097 0.46120057; -0.6852857 -0.60647154 0.545793 -0.7251468 -0.46744436 0.5292142 -0.6982921 -0.7417761 -0.4109808 -0.5780298 0.62040395 0.5106508 -0.59138966 0.5320039 0.4462302 -0.5636203 -0.49637535 0.47395515 -0.6093206 0.5503914 0.4338803 -0.5026244 -0.57260436 -0.43936896 -0.56317866 -0.64428884 -0.49170154 0.45050278 -0.41795337 -0.5885455 0.42833498 0.718921 -0.77292943 0.59990114 -0.75263137 0.6268161 -0.47311676 0.5366741 -0.5470057 0.55243886 0.5075221 0.6154881 -0.71460927 0.7522183 -0.4887938 0.5579028 -0.5350471 -0.6946683 -0.6497725 0.41002518 -0.7841143 -0.6779885 0.5087079 0.6043507 -0.5542069 -0.40695345 0.48137283 0.39206406 -1.3071171 -0.7244317 0.59335035 0.5689793 0.48281112 -0.49165717 0.53434956 0.5865151 -0.73869085 -0.45121673 0.49500713 0.7765717 0.63638717 0.4894784 -0.4655373 -0.60247576 -0.4116413 0.61041903 0.57091683 -0.50496036 -0.6734455 0.46484143; -0.68458843 -0.73514175 0.64381254 -0.7255738 -0.6423048 0.74046814 0.64353114 -0.73131484 0.252459 -0.41271254 0.6451078 0.502125 -0.73282486 0.72125304 0.46614555 0.5196615 -0.61692625 0.54036385 -0.45301974 0.6347306 0.6566931 -0.4242123 -0.5202582 0.5927591 -0.47405383 -0.59068507 -0.7142626 0.7808929 -0.67863137 -0.6634647 0.54490745 0.44966137 -0.74565464 0.8023342 -0.4671124 0.6073519 -0.49994972 0.59553605 -0.44845942 0.72901565 0.65818506 0.5189073 -0.61859477 0.43782303 -0.6648721 0.764968 -0.4284809 0.99883264 0.53242856 0.49779236 0.4918313 -0.3944597 0.55893314 0.5648662 -0.4340896 -0.530383 0.39012885 0.41196898 -0.6915492 -0.723478 0.39593086 0.47971407 0.6259975 -0.7089174 0.03579656 -0.7269275 -0.471704 -0.6251807 0.7328377 0.7036747 0.63695264 0.7658616 -0.66993785 0.6889032 0.3213425 0.58282405 0.722569 -0.5284768 0.80322504 0.5567153; 0.35161453 0.55443907 0.7689307 -0.6301598 -0.5841032 0.61028284 -0.47642767 0.69135237 -0.47645593 -0.65267897 0.8226522 -0.719743 0.50614756 0.4879812 0.5163276 -0.5491037 -0.56825715 0.43238443 0.41869548 0.66390467 0.36748752 0.6415488 -0.797135 -0.50040585 0.36624405 -0.70785016 0.5715363 -0.7987972 -0.61160946 0.76530117 -0.6998161 -0.44849312 0.61649024 0.4109892 -0.513434 -0.6513322 -0.6391201 -0.66046894 -0.77765864 -0.75867665 0.6910532 0.7270737 0.59360427 0.61721605 -0.6850848 -0.57365906 0.64158845 -0.4962493 -0.56407934 -0.3676818 -0.6001553 -0.609289 0.35579473 0.5268709 0.74354273 -0.58773595 -0.7442802 -0.76539755 0.47906002 -0.5717485 0.4562477 0.69904846 -0.74463123 -0.76278836 0.5072052 0.478168 -0.4585236 -0.4839941 -0.6028888 -0.76196206 0.6223917 -0.27570036 0.8016338 -0.61630607 -0.77481514 0.0069522546 0.62552035 -0.7604469 -0.61183697 -0.37977916; -0.40247947 -0.43858588 0.45368636 -0.5565643 -0.4398036 0.60955167 -0.58108974 -0.5867503 -0.63298947 -0.39731377 0.41963276 0.6132411 -0.52562094 0.5609623 0.3817169 -0.3686056 -0.638776 0.51110774 -0.5528596 0.56565523 0.5814763 -0.36560905 -0.703303 -0.48554614 -0.6282932 -0.5320645 -0.7560457 0.45747754 -0.56984025 -0.7505717 0.7228573 0.6967296 -0.39454964 0.5093292 -0.76733357 0.762589 -0.75334597 0.5582938 -0.57732403 0.7177484 0.7026251 0.49865642 -0.46750632 0.54798305 -0.5488881 0.55455095 -0.35651785 -0.66828275 -0.33715758 0.6844256 -0.8511407 -0.7664412 0.46940696 0.7374924 -0.47680897 -0.63897914 0.7987035 0.5298011 -0.03799149 -0.5967986 0.7392038 0.71375877 0.45270243 -0.5241395 0.690394 0.60955155 -0.3915024 -0.6850937 0.6687901 0.42683303 0.7806895 0.53987795 -0.69775623 -0.44964778 -0.6797331 0.55028665 0.6138708 -0.78804475 -0.5480714 0.6125149; -0.5405299 -0.6127375 0.71103925 -0.5752368 -0.54371524 0.4606641 0.96256113 -0.48484737 -0.76811284 -0.43008634 0.54370624 0.6261281 -0.6953043 0.45639423 0.7313554 0.49696577 -0.57325196 0.6884011 -0.39706635 0.4224987 0.64585996 -0.552992 -0.53986865 0.40752447 -0.7176553 -0.73069024 -0.71522146 0.57528734 -0.721111 -0.69709474 0.6803226 0.4715226 -0.42993835 0.66782665 -0.43210104 0.70246524 -0.66845745 0.54657114 -0.43631724 0.75666374 0.4246861 0.41937253 -0.68225753 0.5174094 -0.5773045 0.7475093 -0.6130669 -0.38422158 0.7483428 0.66383547 0.63176644 -0.44599524 0.6481894 0.5579013 -0.58775806 -0.6443698 0.45224953 0.51022196 -0.5425667 -0.43998823 0.60455275 0.59160286 0.42171773 -0.77273834 0.50110936 0.5969319 -0.65695727 -0.4148912 0.52877694 0.6528294 0.7382116 0.74454314 -0.7874442 -0.56792766 -0.577647 0.53613615 0.614233 -0.6509421 -0.50998974 0.5277411; -0.607904 -0.597371 -0.60979843 0.74695456 0.65382695 -0.51294774 0.6522785 -0.6055272 0.5560223 0.77424 -0.7819119 0.47059324 -0.63911873 -0.629786 -0.53712577 0.74734867 0.40300438 -0.6018135 -0.4197674 -0.64980423 -0.44132307 -0.50160784 0.74973327 0.4705184 -0.3122676 0.62851876 -0.66599256 0.7044113 0.70703673 -0.49258482 0.5047946 0.41876578 -0.72588205 -0.8377241 0.6102457 0.7034294 0.7430333 0.5743096 0.4724201 0.5219402 -0.6079935 -0.71761525 -0.498643 -0.4299927 0.8042661 0.443368 -0.5896041 0.44332275 0.78335595 0.52128434 0.75545406 0.6258127 -0.42651418 -0.58280593 -0.7141999 0.6291786 0.39925367 0.5468486 -0.6349837 0.5252458 -0.534557 -0.61972976 0.5989588 0.74207234 -0.38053384 -0.64279884 0.47658944 0.35338804 0.41173735 0.4949838 -0.5061418 0.47894597 -0.74137706 0.71974635 0.7073582 -0.12358263 -0.5809265 0.48951027 0.6436369 0.3574363; 0.5215872 0.71259344 -0.50827026 0.43167242 0.70118713 -0.55292803 -0.6343744 0.65317833 0.5837296 0.7827366 -0.72572684 -0.5066363 0.620209 -0.76380956 -0.43563756 -0.49173632 0.504782 -0.4614196 0.5759344 -0.43181878 -0.724982 0.43906856 0.48987603 -0.7846965 0.42431998 0.70894974 0.76141006 -0.6871951 0.6314754 0.4315008 -0.6909781 -0.43575016 0.66380686 -0.4352991 0.45582688 -0.72723603 0.63708705 -0.7832362 0.5677224 -0.5709656 -0.45321536 -0.63518137 0.5972718 -0.65988374 0.6298419 -0.48835528 0.6937905 0.4359789 -0.77211446 -0.59901303 -0.52772295 0.41661924 -0.50765043 -0.42485633 0.5829522 0.6736011 -0.7124106 -0.70217234 0.74060583 0.4177864 -0.4302658 -0.41245124 -0.47281808 0.61427194 -0.56768125 -0.5582634 0.39869007 0.49717444 -0.65826637 -0.74761087 -0.75270826 -0.70058 0.6812479 0.614942 0.46801427 -0.60698426 -0.6963817 0.51127833 0.7591117 -0.5005171; 0.6414238 0.42746687 -0.5065166 0.60747236 0.6386135 -0.7162565 0.83400023 0.40198046 0.38110927 0.7245379 -0.68011665 -0.6035131 0.539045 -0.725057 -0.56425816 0.50046647 0.41990995 -0.6001005 0.5597623 -0.7671262 -0.45124364 0.718352 0.59458756 -0.24812187 0.42133442 0.73551875 0.721008 -0.67767566 0.47196093 0.737508 -0.5646568 -0.8123268 0.5208685 -0.4857371 0.53282833 -0.69454056 0.68349886 -0.6585489 0.5299163 -0.5725838 -0.6639338 -0.7728161 0.7237348 -0.70572877 0.6965489 -0.7815533 0.55841345 0.55034393 0.033706754 -0.39817813 -0.15456322 0.6905878 -0.4996023 -0.6185215 0.64488214 0.7759517 -0.5787062 -0.6719139 0.7508272 0.77846754 -0.5714335 -0.43588218 -0.56392986 0.6798937 -0.545548 -0.60666525 0.43347645 0.37223628 -0.7466954 -0.45834732 -0.42167115 -0.6172086 0.43893847 0.7452621 0.45314226 -0.5246716 -0.41777408 0.683096 0.5499812 -0.44572264; 0.71700335 0.78967506 -0.37055942 0.48711473 0.6003142 0.40516922 -0.6067813 0.728053 -0.46499908 0.8871477 -0.76286477 -0.5858229 0.630952 0.6711905 -0.20890665 -0.5896571 0.5329443 -0.4322023 0.6700866 0.6941903 -0.7323756 0.6999288 -0.6927971 -0.58622074 0.81497186 0.3389041 0.8027206 -0.7195709 0.61988086 0.5362194 -0.43332916 -0.6406336 0.7464125 -0.46630096 0.37732592 -0.3712041 -0.08484407 -0.49737728 -0.93494743 -0.68924206 -0.66538566 -0.61563176 0.5429079 -0.13024518 0.0630318 -0.5063984 0.74547774 -0.80209804 -0.63871557 -0.7099485 -0.5497255 -0.83028007 -0.6000515 0.8777068 0.68331534 0.81833494 -0.025081297 -0.8302293 0.36165166 -0.582755 0.87724775 -0.6264201 -0.4631445 -0.47282556 0.87310857 0.50863993 0.568168 0.16625108 -0.30422026 -0.845314 -0.33909407 -0.64740837 0.49022922 -0.60690576 -0.46164873 -0.02420708 -0.8456681 0.6619358 -0.9260931 -0.78962535; -0.5130948 -0.5037594 0.7674622 -0.7904228 -0.7807565 0.63468623 -0.50492823 -0.48721412 -0.4668496 -0.6675997 0.5811131 0.5626543 -0.5656863 0.6092517 0.49260685 -0.6171193 -0.5635268 0.7875166 -0.50196147 0.4875191 0.58495516 -0.5068215 -0.5247068 0.6972402 -0.47090977 -0.71945494 -0.5935319 0.7720568 -0.64397526 -0.66844714 0.6824664 0.76337636 -0.5974146 0.5600265 -0.54999053 0.69627327 -0.52375704 0.47770855 -0.45767942 0.55361235 0.44651857 0.62420976 -0.44068024 0.41085392 -0.4211855 0.63320255 -0.49787202 -0.5660542 0.5672996 0.6294986 0.70924693 -0.55657923 0.4655367 0.4201044 -0.6276234 -0.5539301 0.7444267 0.5099498 -0.7200328 -0.42570254 0.4677587 0.4594085 0.48990843 -0.66241604 0.7174134 0.530127 -0.5077095 -0.4397984 0.6091512 0.58093995 0.62973034 0.49446985 -0.5582844 -0.47619826 -0.4958438 0.6330181 0.6140311 -0.47506374 -0.75900507 0.7573009; -0.64581096 -0.8144573 0.5575015 -0.54863447 -0.5645119 0.44311103 -0.4519984 -0.7384831 -0.614949 -0.6282847 0.58236235 0.537028 -0.52008456 0.44368827 0.6729756 -0.34901947 -0.7028135 0.7086537 -0.7610746 0.3982212 0.5006046 -0.58182585 -0.74176365 0.46291497 -0.50342995 -0.4083937 -0.5149969 0.73302233 -0.52914333 -0.58987725 0.70663875 0.48149934 -0.48243582 0.69238067 -0.47136807 0.48928604 -0.5001901 0.65223604 -0.67759687 0.69798094 0.73765326 0.5429088 -0.7798152 0.73549944 -0.4836001 0.4682408 -0.6758199 -0.39033318 0.39384294 0.7318803 0.80298 -0.5141525 0.6570296 0.3981041 -0.5904048 -0.6935112 0.6162221 0.6904038 -0.39228064 -0.5714592 0.6631959 0.63689363 0.7568332 -0.7635997 0.36392683 0.50589085 -0.44173646 -0.751198 0.3437147 0.6520424 0.5865462 0.6213098 -0.64498687 -0.5685378 -0.51569915 0.616684 0.7133253 -0.4926648 -0.71946394 0.4783967; -0.2290656 -0.61503863 0.54162335 -0.65411246 -0.7606045 0.48887387 -0.6616243 0.6883391 -0.6802862 -0.43107057 0.74814606 0.4525411 -0.7329878 0.56129897 0.77949244 -0.62879556 -0.5019606 0.5154374 -0.506403 0.67661434 0.6523377 -0.5517642 -0.55206496 -0.5520964 -0.49140164 -0.5899702 -0.3917931 -0.48482677 -0.64464444 -0.50501484 -0.81599694 -0.6729057 -0.7112573 0.38775057 -0.5099984 -0.51389223 -0.821733 -0.82081974 -0.39428797 0.41393697 0.67201793 0.71112984 -0.75057346 0.67673683 -0.5450685 0.60149217 0.6396461 -0.6515871 -0.7376901 -0.05428941 -0.46155822 -0.47688794 0.44680658 0.607703 0.26521918 -0.7083132 0.199083 0.1692569 0.48766765 -0.47742203 0.76233417 0.58498603 0.41626242 -0.732083 0.40079582 0.5580584 -0.800847 -0.53851783 -0.14238487 0.7633168 0.4204253 0.7017298 0.11408613 -0.60506546 -0.66553414 0.34382862 0.5865621 -0.38590702 -0.4208407 0.5666943; 0.47381705 0.6247923 0.042400982 -0.89764 -0.79636335 0.37819585 -0.4794085 0.8254255 -0.6279009 0.018818352 0.30056822 -0.66721976 0.7063276 0.6019289 0.6102562 -0.5324322 0.14314683 -0.32772028 0.6803259 0.5421468 0.72837347 0.6282301 -0.4980442 -0.71278185 0.739374 -0.67194307 0.7160559 -0.49752334 0.841964 0.413885 -0.6545006 -0.407146 0.77554005 -0.106402814 -0.873092 -0.6788505 -0.4434398 -0.3844286 -0.6660078 -0.66961 0.71411484 0.7778287 0.5101304 0.9709764 -0.7864121 -0.31260693 0.8126611 -0.660643 -0.4492447 -0.5590054 -0.609914 -0.55739886 -0.3018813 0.79960024 0.6784009 0.8016895 -0.04481481 -0.7904094 0.71993464 -0.4747299 0.6342143 -0.4039304 -0.41423967 -0.64485675 0.6527573 0.69091856 -0.19577985 -0.36283627 -0.34426144 -0.7125266 0.81152385 -0.7485485 0.47427282 -0.73690206 -0.6001692 0.19178227 0.09500286 -0.9201808 -0.82460535 -0.7841754; 0.8115311 0.55810404 -0.47379273 0.74991024 0.46599537 -0.6560481 0.64981335 0.8454506 0.82401353 0.6063987 -0.5497454 -0.46382025 0.8176042 -0.5887143 -0.79319835 0.5573538 0.47140488 -0.5783787 0.693916 -0.599955 -0.3849945 0.50883305 0.51174307 0.8709303 0.71695614 0.6728329 0.5919955 -0.54283637 0.76542133 0.5186042 -0.5601366 0.9997982 0.58200294 -0.40963796 0.74098665 0.88189226 0.50512856 0.5048181 0.67296255 -0.3837791 -0.73582655 -0.5795777 0.7983949 -0.40407398 0.46326727 -0.6548348 0.8399656 0.58960074 0.86895543 -0.6693522 0.66044503 0.47163215 -0.48107317 -0.43927988 0.5096015 0.44575754 -0.5047362 -0.77420366 -0.2429207 0.7587739 -0.77148044 -0.43922675 -0.48509493 0.5092918 -0.5004572 -0.6080804 0.5420006 0.54882973 -0.37791735 -0.5628504 -0.39665535 -0.45299277 0.34652954 0.73942274 0.786679 -0.36111015 -0.64993215 0.5200295 0.49309817 -0.7633694; -0.80403906 -0.80527735 0.43029967 -0.42560807 -0.38627836 0.746283 -0.54805815 0.60098815 -0.52714705 -0.55318826 0.7179793 0.72061175 -0.6590008 0.6089998 0.7583467 -0.60196507 -0.63849735 0.44204155 -0.46225104 0.408918 0.46554706 -0.62037545 -0.65763515 -0.7574467 -0.82107455 -0.6822289 -0.52230173 -0.74157333 -0.61171615 -0.35935155 0.19674364 -0.55534774 -0.7395924 0.5551959 -0.73880965 -0.74940574 -0.6745781 -0.75302136 -0.75312585 0.37068105 0.71255124 0.6385165 -0.5731935 0.5142694 -0.6326424 0.5884365 0.6456101 -0.72409034 -0.83659035 0.5855671 -0.40040195 -0.7560883 0.4247802 0.72096854 -0.52910745 -0.76211965 0.22025883 0.46006912 0.42017323 -0.6123689 0.5020144 0.56930566 0.42711028 -0.6011506 0.45388138 0.7381557 -0.8369032 -0.6491214 0.26560807 0.62336916 0.40094826 0.5485309 -0.57484037 -0.60165113 -0.5301333 0.54832363 0.78609216 -0.69295794 -0.36815357 0.5790562; -0.45830667 -0.64076537 0.44577748 -0.72502613 -0.5359381 0.76761454 -0.23093468 -0.72322005 -0.68332547 -0.64899004 0.6029623 0.4712591 -0.5074122 0.6427902 0.63982266 0.42412207 -0.74587256 0.6742838 -0.51492476 0.7329688 0.7146884 -0.6708507 -0.6698314 0.37319085 -0.5035777 -0.76628965 -0.5480951 0.59058446 -0.7138986 -0.58710104 0.44124216 0.43302086 -0.7571191 0.46606266 -0.6094668 0.65198493 -0.5939697 0.4887377 -0.663436 0.59601134 0.41505155 0.5433518 -0.77431697 0.65042984 -0.40441418 0.76642907 -0.48608392 -0.36606506 0.65713835 0.7459507 0.7250457 -0.513226 0.4838158 0.48049167 -0.4986609 -0.5251932 0.34901226 0.63064986 -0.42124483 -0.5256079 0.5080532 0.65028286 0.43110326 -0.65541875 0.2536726 0.48035207 -0.6398569 -0.66790056 0.44712263 0.7181072 0.67972636 0.45628896 -0.4955428 -0.5963564 -0.526238 0.66028273 0.7164377 -0.7416255 -0.42894012 0.7410655; 0.81687814 0.51427865 -0.705276 0.45369837 0.50233155 -0.78780276 0.43858072 0.39448458 0.4772287 0.76196676 -0.8338123 -0.83172476 0.7796681 -0.46451938 -0.66874105 0.66841596 0.68336964 -0.6450679 0.56313777 -0.73144567 -0.61294067 0.7141838 0.76746523 0.47242364 0.45839575 0.729432 0.6698675 -0.13038619 0.60189956 0.5463554 -0.30468732 0.7734431 0.5493224 -0.6276879 0.6614601 0.52241576 0.8264741 0.58720464 0.48512372 -0.67899746 -0.61966157 -0.5190802 0.5805869 -0.58204764 0.72185713 -0.6265195 -0.48625192 0.451703 0.47349614 -0.75521135 0.71999586 0.7284308 -0.44207576 -0.6123317 0.4984306 0.6966976 -0.18434574 -0.6769648 -0.4364763 0.58168733 -0.44207114 -0.7207211 -0.35264027 0.48373473 -0.40736693 -0.50952595 0.5470102 0.7468494 -0.37470967 -0.6015503 -0.6362703 -0.5364042 0.5049444 0.37954083 0.5925452 -0.55139506 -0.7561056 0.7315866 0.4708458 -0.7897807; -0.45912027 -0.55861896 -0.5377502 0.48735848 0.48029754 -0.75848454 0.49754676 -0.7789963 0.45146194 0.70105374 -0.6118557 0.5929932 -0.70259154 -0.44699618 -0.7643646 0.62965155 0.7654578 -0.54750866 -0.40827382 -0.7205172 -0.7307601 -0.6323317 0.49984524 0.75582194 -0.30165163 0.7565943 -0.670798 0.7222224 0.75062114 -0.48550388 0.6133595 0.5170534 -0.39207327 -0.7657089 0.73034036 0.44692466 0.4512844 0.6475077 0.5206978 0.29855168 -0.6071398 -0.7777127 0.040068902 -0.6122231 0.46088016 0.40127325 -0.6401293 0.49942717 0.77071697 0.49042422 0.6701628 0.75940853 -0.76347125 -0.8120164 -0.6582563 0.52756333 0.31247497 0.6556244 -0.45612696 0.6608442 -0.704357 -0.7623692 0.3466013 0.7594111 -0.26793554 -0.60158414 0.514092 0.5315114 0.6316859 0.44644347 -0.5716475 -0.43555114 -0.48353082 0.663227 0.40211046 -0.39416718 -0.49117592 0.61751825 0.4538126 -0.3884187; -0.7203474 -0.4742535 0.40291008 -0.77384406 -0.6882454 0.4286166 0.41905698 -0.4164779 -0.4959517 -0.4482803 0.5793874 0.64446664 -0.37716734 0.4050898 0.42017537 0.7221895 -0.50720614 0.52265114 -0.5172251 0.46508318 0.42045414 -0.5574396 -0.65442836 0.751273 -0.6881792 -0.51455015 -0.47812068 0.50151 -0.7469844 -0.6764989 0.44526204 0.6828046 -0.7248095 0.48799968 -0.547862 0.7790824 -0.47427547 0.7689241 -0.7767186 0.56708646 0.42979565 0.4503998 -0.5117782 0.57588416 -0.5714638 0.7385103 -0.3985905 -0.5817612 0.6050305 0.42495063 0.5271617 -0.6428285 0.5258905 0.4879138 -0.41548356 -0.445664 0.50894916 0.7048462 -0.48738846 -0.5168014 0.50864697 0.51087546 0.7686628 -0.6599986 0.88731253 0.5044324 -0.7204233 -0.6254472 0.51240975 0.56355685 0.61960083 0.5241968 -0.5092349 -0.4066093 -0.40412933 0.5892839 0.39092085 -0.6551527 -0.49740317 0.5355714; 0.64400524 0.6325488 -0.43201572 0.7049886 0.5833657 -0.55275553 0.44113293 0.51753217 0.41250938 0.54504883 -0.55764705 -0.6519773 0.4808989 -0.74550164 -0.7123248 0.63969994 0.4889144 -0.7865352 0.58016217 -0.48383972 -0.62401986 0.5426229 0.53466284 -0.01027432 0.61254114 0.6067415 0.72828025 -0.42279878 0.4808085 0.4545842 -0.7605924 -0.6942781 0.5166488 -0.6779358 0.49139813 -0.5475844 0.6481223 -0.38267317 0.79345554 -0.686743 -0.6800757 -0.676554 0.5568468 -0.6191096 0.42784595 -0.52186275 0.62223166 0.4770454 0.3735947 -0.48090935 -0.53448254 0.49947557 -0.72919667 -0.7817474 0.5430613 0.5436855 -0.62079895 -0.70139205 0.5181144 0.729751 -0.5605827 -0.76773965 -0.74650306 0.45554093 -0.4052942 -0.6059635 0.48574007 0.61486095 -0.41158032 -0.6812588 -0.44842413 -0.7137893 0.756276 0.6061147 0.6275659 -0.44859445 -0.6591699 0.5338989 0.48569673 -0.4548984; -0.46011394 -0.6360521 0.765565 -0.5320703 -0.69807965 0.7669012 0.25570294 -0.77119774 -0.6006533 -0.6999505 0.75822276 0.7733816 -0.56286985 0.6245618 0.76461786 0.37178925 -0.57092994 0.7762159 -0.5110196 0.65274686 0.41867518 -0.5758107 -0.6126654 0.6576584 -0.7062567 -0.709716 -0.47812718 0.45906144 -0.5060725 -0.50966084 0.5362739 0.6793503 -0.5622688 0.49834168 -0.4551266 0.55232877 -0.70782894 0.65975463 -0.7180036 0.50140107 0.4147556 0.72594076 -0.65396696 0.52323645 -0.56986314 0.5581635 -0.77247834 -0.62911874 0.70124507 0.73476034 0.60525876 -0.6988029 0.5174257 0.48368418 -0.7198974 -0.75057364 0.48240137 0.53504515 -0.44721037 -0.56586236 0.62653756 0.7501869 0.5129941 -0.52727205 0.4601244 0.5450413 -0.76605016 -0.6452621 0.7371521 0.49428323 0.46902972 0.65595067 -0.7773373 -0.48420843 -0.5962871 0.6508264 0.68557465 -0.50900924 -0.40841925 0.51450014; -0.7185747 -0.74097735 0.7855405 -0.5037856 -0.46548882 0.6664104 -0.5469724 -0.67641187 -0.6769861 -0.4775481 0.51666313 0.66514856 -0.42309523 0.6122459 0.66486466 -0.4811154 -0.48147792 0.4102408 -0.5650181 0.69233394 0.7093449 -0.47981784 -0.41887394 0.8126747 -0.704093 -0.45547834 -0.5881308 0.5891276 -0.64202994 -0.7884464 0.5880459 0.72367924 -0.48194993 0.5149325 -0.45687395 0.5111614 -0.43825322 0.72856236 -0.51837337 0.69334835 0.6006726 0.54043686 -0.39360934 0.62849075 -0.44858262 0.4258884 -0.5045997 -0.7540234 0.8026969 0.72889036 0.5708675 -0.6999944 0.5072527 0.43755698 -0.5214789 -0.50540185 0.67808616 0.7136676 -0.8108919 -0.6887035 0.7492317 0.4309218 0.61781937 -0.40063298 0.5228622 0.5572005 -0.63388723 -0.4879959 0.49323705 0.4035548 0.606723 0.54781765 -0.6855903 -0.501008 -0.5790543 0.5724872 0.76622754 -0.6593783 -0.55044246 0.57091606; 0.7490123 0.6460706 -0.66717035 0.5143709 0.50180525 -0.47052643 0.8031223 0.6816624 0.7182436 0.4997275 -0.62750244 -0.46055502 0.74891305 -0.6481901 -0.4859727 -0.42031348 0.6780289 -0.42843154 0.7858372 -0.5487602 -0.6641375 0.6109269 0.45738026 -0.39642322 0.6897564 0.5629228 0.68971765 -0.49854892 0.57927984 0.50800157 -0.54368746 -0.42261395 0.7156361 -0.6778149 0.4300359 -0.403743 0.777693 -0.4162954 0.63549083 -0.687968 -0.46994036 -0.43489385 0.7322465 -0.76691276 0.50482816 -0.6306006 0.591758 0.41415802 -0.49656293 -0.44182536 -0.58519036 0.5808353 -0.41172427 -0.5348903 0.7891467 0.6810006 -0.47017196 -0.7609889 0.77394956 0.61601233 -0.67022073 -0.49186516 -0.5678838 0.6137015 -0.7225264 -0.54798627 0.75786746 0.5291828 -0.51309353 -0.69607675 -0.62914646 -0.6195038 0.4350012 0.4697572 0.62950826 -0.61946076 -0.44319546 0.52676815 0.7230311 -0.6544924; 0.6271639 0.74653524 -0.6946882 0.42231926 0.75645155 -0.63292897 -0.6429246 0.41935712 0.70364034 0.63804054 -0.4691928 -0.5748718 0.5429781 -0.69142634 -0.7006112 -0.5126326 0.61171395 -0.47958422 0.39967167 -0.45579076 -0.7361369 0.6681882 0.63025767 -0.5758811 0.5273313 0.534306 0.7063873 -0.6124901 0.78452253 0.4223671 -0.4917072 -0.6881864 0.5053201 -0.48770317 0.6980109 -0.7607912 0.6120746 -0.70712084 0.46277043 -0.6719765 -0.5328928 -0.68992674 0.60602003 -0.41622576 0.6821368 -0.4370587 0.72926074 0.54769874 -0.6420165 -0.501968 -0.6271174 0.43528008 -0.63480157 -0.42427328 0.45586777 0.6544836 -0.5525082 -0.58321846 0.51564926 0.6482625 -0.5964318 -0.3994264 -0.71819407 0.7058626 -0.5532858 -0.6343729 0.7831168 0.71451026 -0.48473954 -0.485859 -0.49277845 -0.42393583 0.4167181 -0.25129646 0.4155117 -0.4283104 -0.5333372 0.63467944 0.49853233 -0.50089246; -0.49580452 -0.4913016 0.4882639 -0.4390719 -0.6742535 0.64011383 0.5566604 -0.7758027 -0.76183254 -0.5692873 0.65417427 0.53190285 -0.750353 0.75198734 0.46932477 0.77673936 -0.61195564 0.5097788 -0.5483457 0.70969296 0.72193146 -0.67194587 -0.63530016 0.5333259 -0.5012798 -0.5516094 -0.45580837 0.5223195 -0.6961388 -0.66291547 0.6522376 0.4247888 -0.45274815 0.6087292 -0.45196033 0.5604268 -0.50839466 0.77278215 -0.46559596 0.6172589 0.4315868 0.4246365 -0.77697086 0.61468923 -0.6027152 0.4128411 -0.65301514 -0.49099058 0.70739377 0.65807575 0.42307046 -0.4752467 0.7626807 0.72357666 -0.7275809 -0.66492736 0.43550304 0.5239739 -0.7516519 -0.45515946 0.5168924 0.5691612 0.50684524 -0.55896693 0.6235511 0.6772005 -0.5194574 -0.5761043 0.68262535 0.7612263 0.44899625 0.7220767 -0.6441954 -0.73355985 -0.5581813 0.48277876 0.61784494 -0.7076788 -0.65521526 0.74581754; -0.5354135 -0.57796574 0.40407494 -0.50571984 -0.66320413 0.61954814 -0.7027095 -0.41468346 -0.42491844 -0.656797 0.54651964 0.41359022 -0.63257354 0.4080605 0.6445781 0.79814583 -0.42021328 0.5511511 -0.7626873 0.46405944 0.6167944 -0.5193359 -0.62048924 0.44346854 -0.7032527 -0.4127183 -0.5068852 0.44831204 -0.67406964 -0.7719903 0.5296059 0.5195193 -0.6797156 0.7819306 -0.4825128 0.42362964 -0.72580075 0.43661302 -0.757247 0.75573754 0.76782423 0.5415262 -0.6919145 0.7282038 -0.5842177 0.4827297 -0.6260751 -0.5003492 0.5947166 0.4580627 0.76342034 -0.61055917 0.6378463 0.4580995 -0.7720315 -0.5559347 0.6939101 0.7266111 -0.45374608 -0.53414667 0.45066908 0.4866623 0.55074036 -0.5606786 0.5555425 0.5781999 -0.75208265 -0.6979859 0.45170233 0.716053 0.42729402 0.49554902 -0.70997953 -0.64966685 -0.5243841 0.4235791 0.4033695 -0.41812012 -0.44737288 0.682129; 0.6934498 0.016926005 0.494422 -0.47724426 -0.68888867 0.4748957 -0.69100064 0.461435 -0.4446164 -0.6616744 0.37939733 0.59662455 0.58926606 0.67606485 0.38250214 -0.54537374 -0.44046804 0.6328967 -0.38049313 0.75195223 0.52550995 0.9167761 -0.39118385 -0.40638542 -0.33290064 -0.61286294 -0.27022123 -0.72658765 -0.40271056 -0.70550084 -0.49174422 -0.6618328 -0.4847443 0.6328845 -0.6577823 -0.6325428 -0.65061915 -0.3774332 -0.42684534 0.7283052 0.79254633 0.65440005 -0.60250133 0.7317581 -0.48173675 0.5510301 0.44823068 -0.6273137 -0.59873927 -0.6303313 -0.8103666 -0.7295544 0.713258 0.5316818 0.68696165 -0.3880576 0.03922267 -0.88605285 0.5108022 -0.7174125 0.78440917 0.46129695 0.48370537 -0.52782214 0.63391113 0.41041636 -0.6966486 -0.5156514 -0.122568324 0.53504896 0.50247234 0.47557154 0.56880856 -0.54720795 -0.5110708 0.6339962 0.63037163 -0.67348796 -0.679352 0.47778758; 0.82448584 0.564231 -0.42280346 0.39994323 0.6216598 -0.68756485 0.5501539 0.60074854 0.83780384 0.44367284 -0.7839594 -0.46462533 0.7396185 -0.6445152 -0.5288107 0.7531395 0.59126645 -0.52057225 0.6614515 -0.5749915 -0.52154756 0.48577136 0.7959281 0.67453784 0.5711225 0.48900148 0.63637197 -0.380706 0.5805747 0.60756195 -0.54688734 -0.5145564 0.56914634 -0.4609009 0.50833875 -0.480473 0.7972348 -0.3294947 0.7099115 -0.4828751 -0.64212364 -0.60623205 0.6668351 -0.429263 0.3900902 -0.6054649 0.7491617 0.5193705 0.9880971 -0.55301505 -0.7302541 0.44203314 -0.59784514 -0.64192885 0.56683356 0.6224931 -0.3259889 -0.68566966 0.735192 0.48007596 -0.69459313 -0.485805 -0.4835059 0.6073897 -0.4283663 -0.7710573 0.57445055 0.6688029 -0.32972473 -0.5590384 -0.49036404 -0.43160617 0.5943676 0.46492547 0.65312934 -0.4555469 -0.50424594 0.6571867 0.6380992 -0.745814; 0.5068495 0.4981331 0.7308364 -0.69284904 -0.64965826 0.54484624 -0.712767 0.6767095 -0.54666823 -0.5641817 0.5371733 -0.4598885 0.526579 0.68645424 0.6300138 -0.5825847 -0.57113725 0.70772874 0.6201576 0.5068411 0.75509787 0.70166737 -0.47382113 -0.7157406 0.06106179 -0.8169083 0.5172743 -0.537039 -0.63402236 0.3988494 -0.26036897 -0.4188067 0.6354555 0.52812815 -0.73082286 -0.6860549 -0.33636752 -0.6207354 -0.67287546 0.040742412 0.709817 0.543826 -0.48424825 0.7851647 -0.74486405 -0.0063155736 0.45212674 -0.6104756 -0.46217343 -0.7579245 -0.5366486 -0.47806716 0.86949706 0.8389004 0.49904275 -0.62410295 0.06562123 -0.5862847 0.69705397 -0.5539287 0.70743 0.71561944 0.11689746 -0.7017735 0.05584021 0.3681322 -0.3019514 -0.33325276 -0.33798087 -0.43164265 0.7265303 0.4080072 0.39111996 -0.48963526 -0.49716854 0.43653667 0.40230364 -0.8187629 -0.8264316 0.3694604; 0.6141437 0.7233408 -0.665923 0.5618151 0.6653875 -0.6132292 -0.7482861 0.42945567 0.62429947 0.719787 -0.43403473 -0.65239495 0.42921486 -0.4128154 -0.70899403 -0.4151039 0.46902704 -0.7205633 0.7499707 -0.46277848 -0.44611388 0.55659413 0.7041833 -0.68025166 0.50546324 0.5664412 0.5016351 -0.7748776 0.47347942 0.4157154 -0.45237845 -0.53953934 0.6789089 -0.61119866 0.59059876 -0.6688471 0.6487596 -0.5595367 0.75376856 -0.7137567 -0.5147424 -0.55116606 0.59820306 -0.589287 0.5297307 -0.62714547 0.78955704 0.49835894 -0.7494439 -0.60159045 -0.74797034 0.63934517 -0.58783287 -0.50682676 0.5534546 0.5298676 -0.64135736 -0.4166842 0.4470079 0.7323842 -0.74382997 -0.47490153 -0.57126135 0.70186716 -0.5515854 -0.70621455 0.77239347 0.7366918 -0.5940967 -0.66603124 -0.68513334 -0.7610661 0.6677327 0.61780196 0.7793533 -0.477794 -0.73683155 0.6490929 0.5467775 -0.7116433; 0.6226288 0.66961384 -0.41684252 0.68485 0.4059214 -0.72166014 0.76340735 0.64268184 0.48539853 0.7676697 -0.6681606 -0.6677854 0.7357532 -0.6165109 -0.48563933 0.58502907 0.51277554 -0.49167538 0.7455067 -0.7486912 -0.7761229 0.5656165 0.40223527 0.42434007 0.5451744 0.7737114 0.46080709 -0.5465638 0.70879734 0.74102384 -0.660266 0.43675405 0.63957405 -0.752401 0.5345662 0.43613476 0.78202164 0.3413151 0.7437423 -0.53290546 -0.5724348 -0.4567915 0.43103513 -0.4328907 0.4791625 -0.7723042 -0.5798185 0.60385406 0.55711263 -0.71921545 0.43790284 0.5620453 -0.730128 -0.6143929 0.58469 0.5018953 -0.53144777 -0.67159307 -0.82720715 0.39665762 -0.45833114 -0.7837824 -0.5538487 0.6347969 -0.19114822 -0.7292229 0.43288207 0.789268 -0.66961604 -0.720791 -0.5603309 -0.657162 0.5744845 0.70266 0.7762896 -0.7189235 -0.6125075 0.6196164 0.5547464 -0.523141; -0.63985515 -0.6519063 0.44240394 -0.6989687 -0.7335995 0.7191051 0.73575205 -0.65842676 -0.5745206 -0.67450124 0.6384482 0.47219104 -0.59199536 0.7087872 0.5422511 0.6259583 -0.5733993 0.69957405 -0.48987633 0.64130074 0.45731238 -0.58538914 -0.4413043 0.65474707 -0.7363931 -0.7872782 -0.4268099 0.65056694 -0.5276361 -0.78621453 0.6297885 0.4497596 -0.6815315 0.4610628 -0.73415804 0.43270507 -0.7117706 0.45662028 -0.52957964 0.6521481 0.6054193 0.44512573 -0.56887954 0.71347696 -0.51333266 0.54999155 -0.6646609 -0.6497147 0.4898225 0.6466152 0.65408623 -0.5381418 0.7735874 0.7198059 -0.7802688 -0.60326266 0.38833562 0.42138618 -0.7144489 -0.44591 0.37832338 0.5627284 0.62150896 -0.7848563 0.39021516 0.55936426 -0.41110933 -0.6537534 0.609811 0.77695453 0.6200601 0.58442694 -0.5104206 0.62224793 -0.52404135 0.43749732 0.7409567 -0.64556444 -0.44372687 0.78935844; -0.4577638 -0.634495 0.6924607 -0.46490628 -0.5679599 0.4475051 0.27153647 -0.5071265 -0.70307285 -0.62147003 0.7063309 0.63340306 -0.72321 0.5942407 0.78172666 0.46826914 -0.66133785 0.55031073 -0.6501224 0.71150005 0.64169264 -0.751135 -0.5970633 0.6332776 -0.65356594 -0.5088535 -0.47475195 0.65769225 -0.46765476 -0.56986463 0.55687886 0.5821802 -0.7411983 0.6939076 -0.743082 0.44633812 -0.77556163 0.6678995 -0.59474194 0.41218796 0.44635767 0.4993331 -0.69469696 0.41442364 -0.7812501 0.4866696 -0.6407652 -0.7987117 0.4431688 0.51391137 0.5824869 -0.7549793 0.6772882 0.56770915 -0.49740526 -0.6054101 0.5174471 0.5732167 -0.7117522 -0.69899434 0.5216867 0.6977061 0.66412854 -0.7308003 0.58301085 0.49651352 -0.58979046 -0.59112734 0.6849887 0.66564023 0.5173672 0.5447383 -0.47721168 -0.7060007 -0.41801804 0.78067434 0.43318495 -0.5998167 -0.7513075 0.6632745; 0.7189154 0.50144154 -0.46765274 0.4215618 0.56653976 -0.66055983 -0.52946633 0.7420058 0.40527463 0.7563989 -0.72319895 -0.74608254 0.45326987 -0.6043037 -0.7584022 -0.6232531 0.7784542 -0.5339646 0.50744534 -0.6618803 -0.61912465 0.5653829 0.7659538 -0.441182 0.6351235 0.6910267 0.6913885 -0.42480522 0.76327837 0.5483177 -0.4123631 -0.44266826 0.7251261 -0.60218406 0.55967915 -0.5004743 0.5246304 -0.4365323 0.6601674 -0.5098009 -0.5498038 -0.46266913 0.43379202 -0.4987891 0.56145924 -0.471016 0.45203125 0.7829393 -0.45724946 -0.63791615 -0.4701919 0.7189519 -0.4964814 -0.77942973 0.48934245 0.56231815 -0.5515038 -0.4286643 0.6990313 0.53239685 -0.5559219 -0.76508296 -0.63541335 0.4601594 -0.6251846 -0.62318337 0.7766609 0.44282046 -0.5135118 -0.7570204 -0.43778718 -0.5996145 0.51705676 0.81900597 0.6859742 -0.79510164 -0.51435304 0.5788419 0.7766134 -0.75548196; 0.53661543 0.6994123 -0.71738005 0.5297336 0.47595733 -0.4813928 0.6817613 0.6701608 0.62917656 0.42340368 -0.70859605 -0.67059135 0.40984958 -0.7084008 -0.5672749 0.32957402 0.74499744 -0.5557179 0.5716349 -0.50040996 -0.4473215 0.69267267 0.6485587 -0.14182378 0.70184785 0.66397345 0.7067888 -0.829342 0.6748031 0.4847558 -0.6626779 -0.9311607 0.6695469 -0.5886982 0.60953027 -0.7048272 0.44994795 -0.6752999 0.546478 -0.56263256 -0.48481315 -0.6572793 0.74038255 -0.46594226 0.6764616 -0.81932676 0.64619154 0.6389621 0.029719332 -0.5501911 -0.28825727 0.67235816 -0.6564233 -0.5052235 0.5877995 0.5350019 -0.6962387 -0.6991507 0.41689277 0.79726213 -0.49534312 -0.6427365 -0.43910652 0.5924883 -0.40850598 -0.660183 0.5700179 0.6839787 -0.69746166 -0.73688364 -0.5624146 -0.68696105 0.6154847 0.5155293 0.6799061 -0.6856142 -0.6740628 0.6154291 0.5653574 -0.6361327; 0.5642794 0.7290867 -0.45193666 0.6254964 0.54244006 -0.58612424 -0.5480322 0.48408738 0.4461046 0.5602095 -0.61976445 -0.4556187 0.5250933 -0.7551368 -0.61345744 -0.7762229 0.45753527 -0.52152014 0.7432709 -0.5402351 -0.76266587 0.7759044 0.5876187 -0.6527463 0.59538525 0.76036054 0.68724394 -0.50405186 0.530702 0.46294913 -0.73892117 -0.6750602 0.4535728 -0.461566 0.7394368 -0.47750014 0.6771003 -0.686883 0.4885453 -0.40407473 -0.5177049 -0.6619078 0.56094503 -0.5345844 0.7560877 -0.42175567 0.6563083 0.43237838 -0.4919899 -0.66158205 -0.6088693 0.62045294 -0.63498664 -0.42698437 0.70165026 0.57369256 -0.5620996 -0.5513208 0.75814694 0.46918663 -0.6870093 -0.42476067 -0.57349545 0.3902919 -0.61609954 -0.7423043 0.6595181 0.47528747 -0.7796563 -0.7621158 -0.48103434 -0.4411107 0.68006504 0.46527886 0.76867294 -0.52851135 -0.68399036 0.68103313 0.59116125 -0.47054413; 0.68681335 0.54265213 -0.5737338 0.6053356 0.6222382 -0.69153094 -0.83640075 0.48413777 0.67322814 0.7121188 -0.4122491 -0.56098866 0.647535 -0.67174625 -0.452918 -0.3327419 0.6539088 -0.7520332 0.76645434 -0.62391853 -0.6853444 0.6424892 0.75114226 -0.7502738 0.56762147 0.6377616 0.6240092 -0.5918826 0.6285786 0.58857036 -0.5517813 -0.68162024 0.47464865 -0.74977076 0.5814587 -0.7470398 0.5686352 -0.4625558 0.48949918 -0.40707505 -0.4643293 -0.42040476 0.75136906 -0.59597844 0.54101616 -0.6052276 0.7526863 0.4769922 -0.38338292 -0.7272684 -0.41624925 0.6136557 -0.4956995 -0.72575617 0.61487454 0.7084441 -0.6402232 -0.4717561 0.7333438 0.5403033 -0.7530968 -0.5545717 -0.5602671 0.5820488 -0.30392545 -0.78584 0.491029 0.45738477 -0.55120665 -0.7725193 -0.4286731 -0.6556023 0.42341813 0.674465 0.70537555 -0.5632699 -0.53338873 0.7039733 0.51128006 -0.7673257; -0.6177986 -0.5541661 0.69409674 -0.4753055 -0.6369143 0.40793368 0.5694698 -0.40313572 -0.4168813 -0.53217936 0.62050927 0.5439741 -0.6025546 0.5342331 0.4551758 0.46509367 -0.5721909 0.41453072 -0.4316661 0.7336134 0.60149497 -0.58326864 -0.72001505 0.50623995 -0.7406198 -0.7487059 -0.5452158 0.5512393 -0.52168095 -0.7471724 0.72144353 0.63180935 -0.43104622 0.7539203 -0.5374839 0.70903736 -0.73166084 0.7454417 -0.5396301 0.47442278 0.40675458 0.5002272 -0.61713684 0.6430959 -0.4327085 0.67409575 -0.46830785 -0.42259914 0.6795739 0.6956535 0.48278117 -0.42438743 0.39804414 0.5231329 -0.6543936 -0.5737237 0.47293955 0.5152745 -0.598431 -0.4993235 0.48289716 0.41922256 0.47949636 -0.53972006 0.7801911 0.26664463 -0.5086221 -0.4376837 0.44119865 0.6315073 0.54278725 0.40122142 -0.6001287 0.5620072 -0.3086968 0.51858056 0.6507146 -0.69732666 -0.7106672 0.68023527; 0.52939457 0.56079686 -0.40083787 0.6872562 0.47981888 -0.70522445 -0.37714648 0.43069017 0.7529621 0.43168986 -0.74539596 -0.7099289 0.47486582 -0.54369926 -0.6646724 -0.6966728 0.75062555 -0.55021286 0.67386067 -0.45754233 -0.49360174 0.6962002 0.64157516 -0.7668447 0.43071336 0.5618419 0.76183367 -0.7716419 0.6244352 0.6502853 -0.46137378 -0.60684764 0.42327386 -0.49761072 0.70916516 -0.7473383 0.7520767 -0.543978 0.39991483 -0.46374437 -0.7661904 -0.5490489 0.47827944 -0.4588473 0.40630937 -0.46305785 0.63572794 0.44975403 -0.5486494 -0.43543828 -0.6988211 0.4740041 -0.40693122 -0.44754374 0.44975305 0.5329485 -0.6737536 -0.7880516 0.48926732 0.4542587 -0.46827012 -0.75303304 -0.65163845 0.45027208 -0.5425465 -0.7344976 0.6415143 0.57035345 -0.5351432 -0.780051 -0.4041567 -0.63375914 0.61579514 0.72543716 0.7055326 -0.43540612 -0.7874388 0.4220111 0.5955396 -0.7817505; -0.4333817 -0.64656955 0.79656404 -0.6855775 -0.7022966 0.6424018 0.747532 -0.62179786 -0.88209003 -0.6480225 0.51348114 0.57478815 -0.4273099 0.5672224 0.6040616 0.43133605 -0.67301905 0.57603973 -0.4004456 0.42869312 0.73534846 -0.7877729 -0.563105 0.46105754 -0.43109533 -0.5450932 -0.45865035 0.54889417 -0.43146548 -0.70933646 0.7355653 0.42253104 -0.4602886 0.5319831 -0.51211387 0.6603701 -0.7480524 0.5188087 -0.54301345 0.60912126 0.5866088 0.66557646 -0.7384763 0.46821743 -0.77690583 0.7352647 -0.65903026 -0.42698196 0.6174586 0.6221063 0.7909457 -0.42959702 0.5422491 0.5162448 -0.64507324 -0.76486135 0.6512246 0.673002 -0.5928561 -0.7823882 0.5951948 0.5927782 0.6776657 -0.78717345 0.35559672 0.8251931 -0.43620342 -0.56664175 0.57262105 0.49007308 0.5218123 0.6783692 -0.76814365 0.86950755 -0.6023181 0.62548095 0.49050683 -0.5732977 -0.7087598 0.5294592; 0.787655 0.5622905 -0.43369558 0.5655801 0.763499 -0.6680373 -0.712404 0.5639712 0.84969866 0.57484114 -0.7814402 -0.41537744 0.5880138 -0.56475294 -0.641897 -0.61682504 0.41602132 -0.46012267 0.64738816 -0.6865152 -0.6672645 0.5101069 0.680848 -0.38110754 0.4961655 0.4162956 0.49469307 -0.41837347 0.6441457 0.5679119 -0.5263917 -0.7654932 0.49505767 -0.6262959 0.70958257 -0.44037005 0.7262593 -0.5595492 0.5536358 -0.61515677 -0.5311289 -0.5291402 0.69336945 -0.4902073 0.5000318 -0.4290268 0.6053344 0.28436145 -0.537545 -0.44008666 -0.53639305 0.5596113 -0.6031483 -0.5358999 0.7534693 0.5363567 -0.51613903 -0.4946054 0.6742642 0.69536453 -0.6190335 -0.5156988 -0.7477418 0.8325082 -0.7786858 0.017428957 0.6069543 0.7802451 -0.5231337 -0.7811567 -0.43841922 -0.77671546 0.6048967 -0.4928934 0.5716286 -0.415271 -0.6721527 0.50799847 0.4925966 -0.51859516; -0.7652426 -0.4028811 0.65382504 -0.57003874 -0.6814161 0.5294677 -0.43385768 -0.71853685 -0.7895205 -0.50647855 0.4892797 0.5789085 -0.55858606 0.6181313 0.7324326 -0.6906031 -0.42269412 0.777433 -0.4461636 0.5985629 0.6873824 -0.5439088 -0.5467932 -0.33182782 -0.58250946 -0.4540123 -0.62311447 0.63784873 -0.46152553 -0.6216904 0.45851368 0.620283 -0.39914685 0.6870665 -0.46806332 0.5870311 -0.77851385 0.5683701 -0.47053698 0.65568906 0.49522102 0.44487864 -0.53250873 0.6523722 -0.5830975 0.5006349 -0.5185097 -0.6241461 -0.61346453 0.620949 0.4569851 -0.75925314 0.5485012 0.45114058 -0.50697917 -0.63939077 0.56408876 0.41401333 -0.7827754 -0.41518044 0.50235546 0.6728098 0.46154937 -0.677369 0.5704463 0.76870054 -0.6742602 -0.5993825 0.61362725 0.5924858 0.43134537 0.74392605 -0.4349952 -0.57326496 -0.78350484 0.7756498 0.64633846 -0.5178757 -0.42077222 0.53134406; -0.40056962 -0.62486744 0.6127342 -0.6258179 -0.54869276 0.7466638 0.82243896 -0.7818239 -0.59997916 -0.5759243 0.46178177 0.7507153 -0.4220569 0.40889266 0.7261492 0.31397673 -0.7806848 0.43198335 -0.7778191 0.54681975 0.5363048 -0.6752635 -0.56028694 0.6868108 -0.7084373 -0.46444362 -0.5562283 0.49028677 -0.70442325 -0.54208803 0.43538743 0.71840894 -0.6944861 0.70931476 -0.4950512 0.43027 -0.74266905 0.71940637 -0.64510703 0.46893534 0.7661512 0.7487186 -0.7467392 0.6731773 -0.45056877 0.672085 -0.5727613 -0.4084586 0.38913786 0.7212894 0.62894535 -0.42494196 0.76630247 0.50786287 -0.74858326 -0.44596514 0.6590078 0.52141994 -0.6124484 -0.65238667 0.76076436 0.5691213 0.7299327 -0.41135606 0.65556854 0.5582704 -0.7484763 -0.7709236 0.50832087 0.57103235 0.54390824 0.56933326 -0.61615276 -0.47927186 -0.56281126 0.6648663 0.5859223 -0.52987915 -0.57713103 0.4943949; -0.6179996 -0.39486027 0.49739385 -0.7236826 -0.69405 0.76528883 -0.45874882 -0.5556523 -0.47486845 -0.7303193 0.6217937 0.633103 -0.7125454 0.5371895 0.7089105 -0.33465335 -0.5381308 0.6874858 -0.776543 0.7030575 0.47917554 -0.6260821 -0.749927 0.56059873 -0.5854645 -0.45915014 -0.5792098 0.54993534 -0.405973 -0.4440816 0.69991565 0.5242302 -0.49891078 0.47984925 -0.55573785 0.5778038 -0.6867408 0.63867563 -0.58067137 0.5021668 0.6984429 0.6067668 -0.6160553 0.5824181 -0.73088825 0.79057765 -0.47627357 -0.59068733 0.64041245 0.383928 0.66004467 -0.68641174 0.4592511 0.45701453 -0.5283503 -0.68096906 0.76609397 0.72561765 -0.67671835 -0.48318142 0.58213323 0.5884601 0.49207324 -0.61789036 0.5932251 0.61987495 -0.5356516 -0.4736266 0.74388385 0.7332313 0.70390326 0.7339295 -0.77530676 -0.6021753 -0.73294175 0.50086105 0.7623131 -0.4765154 -0.46479082 0.7732256; -0.5384831 -0.7303721 0.4605918 -0.4419535 -0.42024162 0.4426997 0.69637984 -0.53408533 0.3255056 -0.73442906 0.75227064 0.73815906 -0.57617027 0.7609254 0.66168183 0.66760546 -0.58937025 0.639583 -0.6817864 0.4919065 0.7870682 -0.48653764 -0.6695474 0.4655674 -0.61705375 -0.5721859 -0.47017857 0.49261084 -0.6679099 -0.5993954 0.67256224 0.60553735 -0.47743905 0.5399044 -0.6588162 0.5046635 -0.4918332 0.69718736 -0.44703415 0.62459624 0.46066028 0.7406161 -0.60069376 0.5420878 -0.7594219 0.63609153 -0.785486 -0.30855858 0.7698219 0.7098344 0.5255461 -0.6650336 0.44516736 0.46841753 -0.7328673 -0.74121875 0.42002735 0.56799597 -0.45312604 -0.4114047 0.4196807 0.529934 0.44938147 -0.7444928 0.5497742 -0.6565071 -0.5444834 -0.4967877 0.64215064 0.6176369 0.51610404 0.6207189 -0.50817513 0.52656764 0.5433005 0.48740652 0.73995537 -0.4840816 -0.32184422 0.74637455; -0.3656574 -0.65377104 0.63932145 -0.653324 -0.63611066 0.6024018 -0.7177291 -0.6868043 -0.4963804 -0.49043187 0.54715383 0.73179364 -0.5765956 0.6440304 0.49191958 -0.4825441 -0.77691495 0.6347146 -0.63703984 0.7349331 0.70503896 -0.37580967 -0.6426179 -0.37437797 -0.5815314 -0.7759081 -0.7047317 0.7427878 -0.64988434 -0.68737876 0.76221234 -0.53161806 -0.7082915 0.528883 -0.6269589 -0.38857377 -0.39814657 -0.1550969 -0.60515934 0.8206102 0.50838804 0.6949753 -0.39640605 0.70993173 -0.7692614 0.51646364 -0.08436898 -0.591132 -0.5312691 0.58874875 -0.8048094 -0.824895 0.5152933 0.6067338 -0.6435349 -0.49338827 0.5177821 0.58070654 0.4135758 -0.7071049 0.5209635 0.7071074 0.7550148 -0.5482645 0.515734 0.45754272 -0.7204124 -0.47582957 0.71869946 0.4518462 0.6353175 0.49075952 -0.7270898 -0.6358453 -0.46605757 0.7541686 0.69354564 -0.6813618 -0.49542832 0.54353297; 0.4522349 0.4114116 0.79145336 -0.78455967 -0.6948462 0.6173606 -0.5213271 0.51095647 -0.6850965 -0.8109231 0.53567207 -0.6429976 0.6272575 0.66249204 0.4063549 -0.44907483 -0.43522668 0.57045305 0.417708 0.7508891 0.58265203 0.5469473 -0.7466599 -0.7492244 0.4857625 -0.67494804 0.48526648 -0.69985056 -0.67147046 0.4955256 -0.6735323 -0.55835164 0.62543976 0.6536376 -0.63630813 -0.58080643 -0.42197186 -0.4615385 -0.5246197 -0.59954345 0.6581591 0.84138215 0.3551895 0.65614223 -0.7709754 -0.27737287 0.7115293 -0.66667145 -0.5491906 -0.43351936 -0.671983 -0.6544722 0.40236545 0.825565 0.40511987 -0.46046156 -0.46694875 -0.686281 0.6410878 -0.66987216 0.45406988 0.5877025 -0.57826877 -0.5096003 -0.07808484 0.7264165 -0.527195 -0.45180276 -0.5239638 -0.41793495 0.56544954 -0.45931295 0.5593211 -0.52805173 -0.49514997 0.23810357 0.7218811 -0.7757283 -0.7353257 -0.2929365; -0.5848993 -0.69797176 0.46641043 -0.7559874 -0.78856957 0.54976696 -0.21206167 -0.7688425 -0.6187406 -0.76051307 0.52370834 0.64616203 -0.75117755 0.6479678 0.6285231 0.39094248 -0.6432341 0.43271407 -0.66290325 0.67662334 0.7062536 -0.665487 -0.7054273 0.3310356 -0.80544233 -0.49868718 -0.5497206 0.4511553 -0.77558655 -0.507676 0.74835974 0.58981323 -0.50522983 0.47091204 -0.4724053 0.36143422 -0.44237104 0.48667666 -0.69527173 0.6996991 0.6223717 0.48853737 -0.6203322 0.44045565 -0.51503307 0.61897504 -0.6688396 -0.48992753 0.6903839 0.5908598 0.44694254 -0.746915 0.46433285 0.60238546 -0.6906308 -0.74835867 0.29439276 0.79626256 -0.46885997 -0.689942 0.50932103 0.5908808 0.5559131 -0.6382162 0.24536723 0.688589 -0.70109737 -0.54598045 0.66740775 0.5031676 0.46798643 0.60861677 -0.52292144 -0.53864604 -0.45871535 0.6884115 0.6354955 -0.4841819 -0.6824011 0.5493029; -0.7541109 -0.5948327 0.43364072 -0.653776 -0.5839835 0.57540077 0.7364016 -0.591194 -0.62865835 -0.47968563 0.41408515 0.4403491 -0.622529 0.5954226 0.4480811 0.68236756 -0.70095646 0.6394494 -0.60054684 0.5269528 0.69655025 -0.42793792 -0.4097388 0.51826656 -0.6585766 -0.55313164 -0.74517506 0.6402757 -0.5821653 -0.7430525 0.54066944 0.7226993 -0.73707384 0.5012796 -0.47116616 0.45056322 -0.40579903 0.48234937 -0.560792 0.51645917 0.47486246 0.72666496 -0.6888136 0.5374727 -0.40247858 0.44737643 -0.6447949 -0.4623765 0.6959275 0.50652426 0.6954976 -0.59919345 0.5671343 0.712636 -0.5657083 -0.77985513 0.43201384 0.66779965 -0.6211015 -0.52313614 0.55712724 0.724424 0.62246495 -0.6682412 0.6482531 0.45182583 -0.7318145 -0.6292149 0.781029 0.6394851 0.40607402 0.71875834 -0.5055276 -0.8117456 -0.73033917 0.4131504 0.6285221 -0.40058416 -0.49098203 0.6215609; 0.60516554 0.6099043 -0.69140536 0.55988234 0.6504218 -0.80188125 -0.5771446 0.44527096 0.4945049 0.6914338 -0.5511236 -0.79073644 0.8153454 -0.5303084 -0.74052125 -0.52745324 0.46574947 -0.76452017 0.7217503 -0.7017643 -0.65011156 0.48209038 0.46130756 -0.53402126 0.5617317 0.4402528 0.58563393 -0.45173052 0.48391303 0.58266395 -0.42173615 -0.49396133 0.5222106 -0.64610106 0.51969993 -0.7249033 0.77248794 -0.4856333 0.7629928 -0.4597685 -0.7047203 -0.7844331 0.4364428 -0.45846975 0.66593885 -0.5394506 0.824858 0.54133385 -0.53464043 -0.55052423 -0.78933984 0.3926957 -0.4537917 -0.7424167 0.7870992 0.7236396 -0.3210572 -0.66915447 0.49739835 0.7466381 -0.43207645 -0.7796504 -0.6313866 0.5006883 -0.22662504 0.1884914 0.7812592 0.5685019 -0.4244103 -0.53449625 -0.5327803 -0.45012987 0.5802187 -0.7016832 0.68429846 -0.46379548 -0.75959814 0.6701823 0.31144875 -0.5833749; -0.5454493 -0.670564 0.4884618 -0.69280666 -0.585348 0.4388266 -0.6889004 -0.44360134 -0.56180525 -0.4682593 0.6751027 0.44421077 -0.66914487 0.44883165 0.74592435 -0.54741836 -0.6138739 0.5301171 -0.4420565 0.45947763 0.63294965 -0.6071926 -0.44880104 0.7638054 -0.54460245 -0.51290256 -0.59585416 0.7609194 -0.64845806 -0.6578936 0.665292 0.74066067 -0.67001307 0.41340142 -0.6206833 0.5480655 -0.46113795 0.55033916 -0.6762785 0.43696886 0.76302385 0.48032472 -0.45584914 0.41832396 -0.69757867 0.47343814 -0.52218884 -0.7446352 0.65473914 0.43886477 0.6711293 -0.70405734 0.40963373 0.57525754 -0.4947879 -0.5612276 0.74837613 0.7720253 -0.6186655 -0.5235105 0.59806883 0.64438045 0.5881421 -0.48105928 0.7003068 0.6788602 -0.7264328 -0.74723786 0.6853496 0.5054286 0.69690335 0.75024843 -0.52426577 -0.61810976 -0.72883475 0.60566914 0.6454095 -0.53011197 -0.48817438 0.63803536; 0.42820096 0.62754995 -0.5420206 0.5881448 0.47746938 -0.49138978 0.6116547 0.716092 0.43303245 0.5909132 -0.4040484 -0.4185016 0.68307996 -0.6199786 -0.60795146 0.4785249 0.67272204 -0.7078958 0.4007858 -0.5846817 -0.4083042 0.6853519 0.73616207 -0.8576351 0.55956817 0.57902235 0.5721193 -0.8145003 0.5635081 0.6665175 -0.7884344 -0.7240525 0.75521654 -0.4648578 0.6115547 -0.4694982 0.734837 -0.60053915 0.7246102 -0.41999143 -0.50614923 -0.5985337 0.76810086 -0.48460478 0.6092478 -0.77650917 0.4903769 0.5876063 -0.5764597 -0.44895685 -0.501655 0.48749202 -0.41565764 -0.5393165 0.5777659 0.3943414 -0.8410065 -0.7308964 0.8056514 0.7032756 -0.66064364 -0.6315278 -0.6122364 0.38898242 -0.67698693 -0.45989367 0.42256463 0.68531895 -0.50251067 -0.59650266 -0.415792 -0.5329649 0.51986116 0.7401742 0.5301126 -0.55633557 -0.4784441 0.5067577 0.510473 -0.40897962; 0.7849952 0.5165067 -0.7662452 0.5910849 0.7511948 -0.4177409 -0.7292667 0.6823546 0.6317832 0.7624243 -0.47392178 -0.59048456 0.742521 -0.4324762 -0.64545524 -0.57417035 0.56878495 -0.5279028 0.7143487 -0.5482222 -0.6860916 0.6596594 0.54424685 -0.43266225 0.7600105 0.7766937 0.4170651 -0.4543334 0.6472772 0.55909526 -0.58412343 -0.7487429 0.7149267 -0.6801849 0.41992474 -0.5310576 0.7472832 -0.5108298 0.5987234 -0.67065704 -0.6079199 -0.592041 0.6085999 -0.5835498 0.5291943 -0.4423168 0.6520054 0.6706228 -0.5422107 -0.6364832 -0.5491975 0.53600997 -0.77909476 -0.7149519 0.6291117 0.50736654 -0.41509116 -0.7518894 0.4312036 0.6419561 -0.43913758 -0.67785126 -0.718746 0.4716342 -0.5657328 -0.7169673 0.6694281 0.7252378 -0.43036002 -0.4513791 -0.5993993 -0.53863263 0.45159903 0.5718894 0.5080752 -0.51841015 -0.43963027 0.5249968 0.65198183 -0.6156867; -0.75631136 -0.4425035 0.5518512 -0.70501816 -0.7217635 -0.7277115 0.56472003 -0.6826592 0.75695693 -0.41430557 0.4305252 0.49394193 -0.5496915 -0.83474207 0.14270939 0.44352278 -0.674697 0.3869422 -0.4224835 -0.54878026 0.68477774 -0.4453901 0.47293717 0.7593749 -0.67830896 -0.5844836 -0.5499413 0.7745189 -0.3944879 -0.56082076 0.7309808 0.61024857 -0.63338757 0.43619487 -0.62328124 0.40251866 0.34450233 0.7457144 0.8538487 0.47632062 0.49581054 0.4898881 -0.46278653 0.35568914 -0.51340157 0.596047 -0.523754 0.6041403 0.7204088 0.5176447 0.5528909 0.72062665 0.7332367 -0.47317028 -0.5501 -0.4907089 0.64135677 0.5956399 -0.61030585 -0.083139464 -0.614725 0.7485315 0.48499414 0.45602268 -0.15209107 -0.5994399 -0.47617188 -0.07521546 0.6040811 0.64472574 0.6594141 0.6432443 -0.6673816 0.69102544 0.43304765 0.5268336 0.6923109 -0.37639537 0.5791208 0.43949333; -0.75235164 -0.71877193 0.68883526 -0.7172161 -0.56421435 0.6451585 0.59091395 -0.6825768 0.066428915 -0.58409697 0.6482898 0.7836493 -0.6043261 0.5497815 0.5845641 0.5066822 -0.45551562 0.55909586 -0.62127155 0.44342825 0.56551087 -0.58152014 -0.57243925 0.5366795 -0.53822535 -0.46506068 -0.6179027 0.57423013 -0.7144598 -0.6207676 0.5908347 0.5939487 -0.46264377 0.66853553 -0.750274 0.49249643 -0.72164387 0.4193261 -0.41241467 0.7626435 0.5771026 0.48867005 -0.47606048 0.7226777 -0.45499718 0.50161725 -0.49807066 -0.15502472 0.60877234 0.5005763 0.58267266 -0.5179486 0.6049129 0.5378305 -0.5886051 -0.4833214 0.45191598 0.7048634 -0.45321396 -0.55240256 0.562206 0.71018946 0.54587203 -0.5242727 0.36026043 -0.43539727 -0.5110691 -0.7969802 0.649205 0.59301674 0.7464467 0.5648626 -0.6666706 0.8576634 -0.0026786565 0.4609719 0.5567653 -0.6808151 -0.45299527 0.6802924; 0.51232237 0.5367302 -0.5899687 0.50712013 0.5826542 -0.41129088 0.79390365 0.7493149 0.3947123 0.6672174 -0.38991427 -0.5093764 0.39072794 -0.5312425 -0.4405959 0.5026229 0.63175505 -0.56880635 0.44682673 -0.5193095 -0.44058505 0.69244 0.75281245 -0.9142932 0.47270954 0.43324542 0.5886786 -0.71662533 0.5195215 0.6841846 -0.76668406 -0.5087357 0.74056184 -0.7725964 0.5228771 -0.7364779 0.59569347 -0.7136666 0.49212703 -0.63768256 -0.5994981 -0.7380768 0.46993697 -0.6507655 0.6818779 -0.7796984 0.5347423 0.59645253 -0.58301854 -0.58652633 -0.36292183 0.4101214 -0.51070607 -0.74073553 0.7409095 0.6685804 -0.7183681 -0.71606314 0.79260826 0.46210644 -0.49849963 -0.47993237 -0.76203126 0.5604688 -0.6712107 -0.7445021 0.40695614 0.57608306 -0.7152231 -0.66445696 -0.70805603 -0.49120387 0.46452686 0.59896 0.72095686 -0.47562096 -0.6542411 0.42220542 0.6540465 -0.4383404; 0.4460831 0.66585094 -0.6235058 0.721019 0.5412476 -0.56002367 -0.73189366 0.40073866 0.44102255 0.6301759 -0.71887434 -0.621989 0.6105536 -0.43510365 -0.45552 -0.40424314 0.40445834 -0.49195537 0.7326705 -0.6705955 -0.7246771 0.52175623 0.46403614 -0.4115542 0.6256053 0.46476474 0.4704072 -0.6033355 0.7168188 0.43229982 -0.51996964 -0.7331835 0.78659797 -0.56072193 0.4811365 -0.6417825 0.7571639 -0.5635162 0.4903276 -0.7360458 -0.5375507 -0.64089954 0.7422884 -0.4262073 0.47229657 -0.50412005 0.6280927 0.5233738 -0.7046683 -0.7097208 -0.4949802 0.45010194 -0.7176885 -0.5821253 0.43435535 0.6473989 -0.54516536 -0.55879563 0.5295844 0.40599832 -0.5026619 -0.5310132 -0.57021457 0.7437655 -0.57356334 -0.5368202 0.54212505 0.7575117 -0.6853931 -0.42289013 -0.5209224 -0.6463048 0.49000862 0.50156 0.7445495 -0.5821891 -0.59212184 0.50917125 0.6323807 -0.6385778; 0.49142277 0.6700474 -0.74064577 0.5511096 0.6157062 -0.6960631 0.675289 0.6632945 0.48546892 0.6529356 -0.75338733 -0.5924269 0.62288165 -0.716039 -0.72998875 0.6154967 0.5536756 -0.53668106 0.47851747 -0.7564709 -0.6281244 0.6271179 0.6473617 -0.535031 0.42067814 0.4641104 0.6786469 -0.46930832 0.68507266 0.51346433 -0.75004166 -0.57002515 0.44640434 -0.4236393 0.48408985 -0.52334225 0.70465046 -0.43012667 0.52348727 -0.5577584 -0.6343641 -0.41493353 0.53270155 -0.73224306 0.6231593 -0.70330894 0.48142526 0.43445045 -0.70022684 -0.70741194 -0.40800402 0.42425182 -0.5376959 -0.45140722 0.6945146 0.6275495 -0.55868614 -0.60540974 0.51045305 0.57085085 -0.66115385 -0.5816905 -0.6694103 0.7711129 -0.6494675 -0.43993026 0.68503326 0.6840426 -0.7659873 -0.44431645 -0.44391683 -0.6812618 0.47732943 0.51295906 0.6190193 -0.521389 -0.408177 0.4226104 0.53471947 -0.7427069; 0.51561886 0.522162 -0.6129634 0.64529794 0.42112693 -0.79277426 -0.54920846 0.68670774 0.8444545 0.5468073 -0.77129424 -0.74032676 0.50732803 -0.7639524 -0.47134528 -0.6805901 0.51454234 -0.616605 0.6248653 -0.700757 -0.63359314 0.5046079 0.42370653 -0.7187913 0.736748 0.5925234 0.71301377 -0.71745604 0.60692495 0.7167926 -0.4898769 -0.6742823 0.58733475 -0.65825146 0.6161501 -0.53964806 0.5726206 -0.69042635 0.3969941 -0.5570264 -0.4543553 -0.6828103 0.5049491 -0.66520995 0.41108957 -0.47227257 0.50606495 0.6675777 -0.59458697 -0.6463213 -0.46149167 0.4244399 -0.757524 -0.72470653 0.6190289 0.7410032 -0.3545451 -0.602147 0.47055545 0.6028327 -0.44546807 -0.6263199 -0.44895318 0.7675609 -0.50312144 -0.6122334 0.57003367 0.7818412 -0.5116285 -0.68309116 -0.43701184 -0.41751713 0.6237891 -0.97930706 0.82550377 -0.5761206 -0.71083426 0.61716205 0.6904424 -0.7059403; -0.68547034 -0.5347332 0.7596709 -0.5847074 -0.50573206 0.45455885 0.76195925 -0.5214409 0.50266993 -0.66707075 0.47508478 0.47809464 -0.6226779 0.56370425 0.68174905 0.6220908 -0.6447027 0.70985013 -0.54157484 0.91000205 0.6821017 -0.5168629 0.6228629 0.6884414 -0.43970466 -0.61613405 -0.5970199 0.66251343 -0.40224928 -0.47085002 0.7415028 0.55823493 -0.5248591 0.6120176 -0.7764759 0.50964326 -0.1938206 0.65290135 -0.6944167 0.5113137 0.50069445 0.580899 -0.54767615 0.7094779 -0.49495366 0.5637963 -0.47374752 0.37020695 0.72189146 0.47957283 0.72362006 -0.68408483 0.45664766 0.7138911 -0.5672544 -0.4221273 0.694022 0.57496333 -0.43583506 -0.48269856 -0.23556018 0.68892723 0.72198504 0.5286607 -0.7174485 -0.45258364 -0.6796078 -0.37094167 0.58921015 0.6547467 0.5768019 0.638982 -0.55773604 0.46410236 0.47278124 0.48993683 0.7461496 -0.65789795 0.61589384 0.67201835; -0.6021359 -0.6783402 0.68798214 -0.6929806 -0.689169 0.6563752 0.5823521 -0.71417 -0.5264126 -0.62393445 0.54049605 0.59025705 -0.60349417 0.51166856 0.49322122 0.68879277 -0.52965003 0.41945022 -0.6184817 0.5799782 0.55934244 -0.59260255 -0.66551006 0.601816 -0.7488034 -0.60980463 -0.7162755 0.6917799 -0.5918147 -0.55139595 0.4243615 0.65908706 -0.5685427 0.58167243 -0.73560905 0.5423479 -0.7136155 0.524711 -0.46413705 0.56855077 0.56447375 0.48900434 -0.69147617 0.5884419 -0.48596507 0.76988924 -0.55344504 -0.68134004 0.50704974 0.47101197 0.7429205 -0.55630565 0.66865736 0.5782943 -0.53647774 -0.594182 0.52419096 0.398279 -0.69696677 -0.749333 0.56123114 0.6479774 0.5248567 -0.74229926 0.61276 0.4626189 -0.72982144 -0.7063871 0.66518384 0.6125007 0.5710764 0.54841775 -0.7494466 0.5897822 -0.76297987 0.7606531 0.5434378 -0.49748224 -0.70409584 0.7491516; -0.38226247 0.35648772 -0.7599661 0.7914929 0.615158 -0.6953587 0.5179527 -0.49171257 0.6682338 0.7921192 -0.40097216 -0.6065988 0.0645371 -0.6783434 -0.6431537 0.52652436 0.59513116 -0.7601701 0.49993706 -0.5499217 -0.68278587 -0.54283273 0.6727116 0.5014895 0.3378608 0.4621673 0.48202774 0.592595 0.7708475 0.40537983 0.16007178 0.64393646 0.6421712 -0.5834183 0.75701505 0.45470268 0.5599654 0.41076818 0.7981905 -0.45231074 -0.7710197 -0.6014945 0.51983696 -0.5486128 0.78499144 -0.6751465 -0.5310993 0.65224946 0.65327704 0.66035676 0.72397923 0.5451592 -0.58643353 -0.6214239 -0.696603 0.7644496 -0.28176543 0.6556569 -0.431276 0.78387 -0.61699235 -0.6755272 -0.6623329 0.489062 -0.09229958 -0.5063896 0.49500242 0.38789958 0.086355224 -0.566785 -0.8195347 -0.58524835 -0.22191548 0.5087301 0.66767937 -0.72019494 -0.67020565 0.48082876 0.6901343 -0.43705493; -0.77224416 -0.8357566 0.3736583 -0.5763588 -0.7265458 0.51517767 -0.46230778 -0.76978713 -0.79454553 -0.6537301 0.76545566 0.76163566 -0.4964542 0.42910805 0.5632006 -0.7498051 -0.65152395 0.45318243 -0.76790255 0.49352494 0.56343484 -0.5466819 -0.6679236 -0.29943302 -0.74543285 -0.73824847 -0.7212543 0.39444488 -0.71554697 -0.38730562 0.47813657 0.47931182 -0.8344214 0.73569214 -0.615972 0.5441362 -0.8472579 0.39486927 -0.5619031 0.5615027 0.6842533 0.5426755 -0.6452865 0.66427255 -0.4402513 0.58161163 -0.78405213 -0.65790004 -0.31669334 0.7564653 0.63413036 -0.5545035 0.7125501 0.6284496 -0.66911554 -0.647607 0.541129 0.7934787 0.019345317 -0.4312016 0.55012107 0.5478638 0.5981192 -0.8431813 0.26191652 0.7048485 -0.66150236 -0.48192304 0.65962416 0.48953626 0.72773176 0.5694975 -0.7297503 -0.5418987 -0.6816737 0.43825087 0.7385224 -0.68428165 -0.3557078 0.77774453], bias = Float32[-0.60819674; -0.57788855; 0.7558195; -0.693859; -0.60239094; 0.6076436; -0.5994925; -0.62695974; 0.5907321; 0.5900563; -0.615696; -0.5993879; 0.5859901; 0.5914623; -0.59384066; 0.6251201; 0.616983; 0.5863834; 0.5959243; 0.5518288; 0.57764876; 0.5844791; -0.50390166; -0.5919644; -0.5814051; 0.34004804; 0.58379865; 0.5753396; 0.59455806; 0.48754147; -0.599347; 0.6116193; 0.5805472; -0.6347724; -0.5081205; 0.6114504; -0.5558099; 0.58181196; 0.58379; -0.5982546; -0.60523707; 0.59866667; 0.59500223; 0.55014014; -0.62343687; 0.46401033; -0.58145225; -0.56695926; 0.583666; 0.58574045; -0.5818838; -0.56246954; -0.59659714; -0.5839183; 0.62406385; -0.59639955; 0.58162445; -0.6111036; 0.57498235; 0.5828105; 0.5701362; 0.60049444; 0.50831306; 0.47280294; 0.5660823; 0.59471476; -0.6067937; 0.5887919; -0.5907295; -0.5871602; -0.08249908; 0.6125489; -0.5831754; -0.59873956; -0.5973027; -0.5948021; 0.5153217; 0.57785624; -0.51566494; 0.6411579]), layer_3 = (weight = Float32[-0.60941684 0.77903455 -0.6558041 -0.60660315 -0.42386284 0.5784391 -0.5452084 -0.6705769 0.4550458 0.4746459 -0.57369816 -0.5083726 0.35359612 -0.6533603 -0.38257167 0.64544106 0.6226509 0.32561013 -0.4522245 0.3107697 0.33262643 0.36005154 0.8582719 -0.41410142 -0.33551526 0.51883143 0.33410484 -0.62474185 0.40625027 0.4889317 -0.47447696 -0.85572696 -0.7003617 0.742743 0.80861187 0.61039203 0.7773317 -0.82788134 0.41194716 -0.3235094 -0.5522417 0.48899746 0.35611603 0.46359164 -0.45368594 -0.66389006 0.7765977 0.50259185 -0.6015291 -0.74234766 0.689764 0.7469371 -0.4876422 0.8019231 0.6436031 -0.31460664 -0.6061944 -0.5839 0.34440157 -0.80162436 -0.8121275 -0.13172863 -0.6250239 -0.73512757 -0.65014637 0.38590413 0.5710323 0.403652 -0.5217551 0.7478377 0.7111334 -0.51201195 -0.47736904 -0.4488409 -0.40099394 0.6224836 -0.28939563 -0.566522 0.81912166 -0.619169; -0.36642635 0.6970802 -0.7588632 -0.5160411 -0.67193466 0.5105679 -0.409786 -0.41288054 0.53613657 0.56774193 -0.39288643 -0.40296862 0.5972002 -0.72658885 -0.43643135 0.35992703 0.3343394 0.6063106 -0.8199092 0.75954014 0.69155073 -0.9859696 0.30012986 -0.4504248 -0.6259612 0.6913001 0.6314813 -0.62163234 0.65618503 0.6942597 -0.64612037 0.66031736 -0.6550633 0.5244696 0.37699443 0.41285303 0.58662355 -0.80478567 0.6120455 -0.34481242 -0.44187668 0.33978364 0.3633132 0.6988822 -0.5581826 -0.52014315 0.74385065 0.727354 -0.7123297 0.17369343 -0.33252043 0.55552524 -0.41805562 0.73847675 0.41540122 -0.34262154 -0.8345496 -0.43955654 0.6181014 -0.8607755 -0.5836813 -1.1217937 -0.58249664 -0.4416919 -0.6836124 0.38925594 0.7063333 0.54890436 -0.56490475 0.76883215 0.43521494 -0.8407628 -0.6159585 -0.34883544 -0.47635886 0.76032406 0.0673244 -0.8765356 0.37624866 -0.69761914], bias = Float32[-0.58092946; -0.6013908]))
loss_neuralode(p_trained_moderate)




#----------------------high noise loss collection 

rng = Random.default_rng()
Random.seed!(99)


x1=Array(sol)

x1_mean = mean(x1, dims = 2)
noise_magnitude = 35e-2
x1_noise = x1 .+ (noise_magnitude*x1) .* randn(eltype(x1), size(x1))
#Displaying true data vs noisy data



#-------------------------------------Defining the Neural ODE------------------------------------


dudt2 = Lux.Chain(Lux.Dense(2, 80, tanh),Lux.Dense(80, 80, tanh), Lux.Dense(80, 2))
#Setting up the NN parameters randomly using the rng instance
p, st = Lux.setup(rng, dudt2)



prob_neuralode = NeuralODE(dudt2, etaspan, Tsit5(); saveat = etasteps)


function predict_neuralode(p)
    Array(prob_neuralode(I, p, st)[1])
end


#training data
true_data= x1_noise[:,1:end-20]
### Define loss function as the difference between actual ground truth data and Neural ODE prediction
function loss_neuralode(p)
    pred = predict_neuralode(p)
    loss = sum(abs2, true_data .- pred)
    return loss, pred
end

p_trained_high = (layer_1 = (weight = Float32[-0.57301587 -0.01950864; -0.51684624 -1.0101635; -0.33591402 -0.2304012; 0.38612923 0.32692778; 0.43255183 0.31555378; 0.9794265 0.63946307; 1.2122017 0.5749467; -0.018815229 -1.3592038; -0.71802115 -0.6381112; -1.2140877 -1.1919576; 0.85560656 0.68698215; 0.44715628 1.1491834; -0.61502373 -0.7789568; -0.4744861 -0.5096282; 0.74285656 0.723538; -0.35510787 -0.5740138; 0.29713976 0.23459542; -0.48050174 -0.2159296; -0.41872177 -1.0789784; -0.566576 -0.48316985; 0.7667623 0.96376956; -0.55804724 -0.9196817; -0.9088322 0.43381125; -0.29600748 -0.3283064; -0.76059735 -0.97236806; 0.26059 0.22752964; -0.15994528 -1.7952354; -0.6462827 0.15797617; -0.8691047 -0.89677745; 0.436719 0.1759434; -0.6125249 -0.34114745; -0.6785532 -0.23319998; -0.37912783 -1.2729682; -0.41164947 -0.1897984; 0.40672758 0.40615708; -0.64453965 -0.2833637; -0.7448946 -0.6565921; -0.71277636 -0.21335396; 0.52350575 0.43137845; -0.5600088 -0.12259419; -0.37845644 -0.21400166; -0.40106684 -0.39870086; -0.6684827 -0.93917763; -0.5457631 -0.42418778; 0.4384864 0.47619236; -0.5130368 -0.38777825; -0.6732405 -0.8873155; 0.5704209 0.4789742; -0.31629756 -0.3352503; 0.5393288 0.53289217; 0.66823095 0.94543356; 0.67613846 0.46002924; -0.4678648 -0.40126657; -0.46399662 -0.51235086; -0.6104379 -0.88674754; -0.954235 -0.94741833; -0.73040336 -0.47380492; 0.3231201 1.0745445; 0.54206365 0.26949686; 0.38404217 0.45090446; -0.5992908 -0.52364093; -0.40696037 -0.19202137; -0.53619134 -0.22320464; -0.8502573 -0.69185627; 0.0842963 -0.6067811; 0.6727603 0.5907348; -0.7369267 -0.7352911; -0.69446814 -0.7159637; -0.62850696 -0.43069428; 0.54263246 0.9357348; -0.42007002 -0.30213714; 0.5782237 0.94482267; 0.50945586 0.0835712; 1.5299948 0.48952073; -0.8186597 -0.59157693; -0.67546666 -0.42034686; 0.97130287 0.88174915; 0.429646 0.29644564; 0.5753574 0.5069249; 0.7459299 0.9384917], bias = Float32[-0.7926695; -0.5746118; -0.58080107; 0.5782551; 0.5145581; 0.6693814; 0.5424254; -0.24085706; -0.76840276; -0.411455; 0.7796231; 0.5049642; -0.7695097; -0.6014831; 0.6060046; -0.6316602; 0.7676965; -0.65121055; -0.6265183; -0.6791526; -0.49381718; -0.63097584; -0.7035095; -0.9832314; -0.5981759; 0.6572708; -0.2219124; -0.6323157; -0.70500094; 0.58919406; -0.6197227; -0.9627325; -0.65417093; -0.63792795; 0.44095466; -1.0227547; -0.75218064; -1.035303; 0.6773073; -0.4966977; -0.47516954; -0.48889667; -0.68697566; -0.5942937; 0.60240763; -0.51941264; -0.6206829; 0.5290254; -0.94679135; 0.6352642; 0.5186443; 0.7556459; -0.49328125; -0.63275206; -0.6701638; -0.67490786; -0.6196751; 0.5771486; 0.10351109; 0.5723684; -0.5801651; -0.69847244; -0.38567096; -0.62420696; -0.030881803; 0.70479035; -0.7157777; -0.647653; -0.66403174; 0.67279696; -0.5673289; 0.48974547; 0.57567215; 0.40358993; -0.81660306; -0.7130429; 0.6383919; 0.51314473; 0.6134167; 0.7442954]), layer_2 = (weight = Float32[0.6164216 0.6433549 -0.5947763 0.46026734 0.55982155 -0.403256 -0.78410345 0.49666804 0.4933917 0.6482213 -0.62185323 -0.5715522 0.49954945 -0.5332347 -0.5192268 -0.48038697 0.48241258 -0.4734075 0.54043734 -0.5860366 -0.38468602 0.49863923 0.5207228 -0.46932662 0.40196934 0.5428543 0.46732625 -0.68826115 0.58237255 0.5397815 -0.46538624 -0.4892104 0.625631 -0.5694979 0.51998866 -0.54917735 0.52058744 -0.7246932 0.51797134 -0.7395946 -0.659849 -0.56401455 0.44305193 -0.42006066 0.45688823 -0.7515336 0.5558089 0.5386014 -0.53929335 -0.73199755 -0.7970633 0.69829184 -0.56045336 -0.62683 0.71181095 0.60883564 -0.46030194 -0.6840903 0.6061013 0.4081621 -0.7007706 -0.6809236 -0.7893891 0.49132636 -0.79927933 -0.53017586 0.63320696 0.49548423 -0.52439713 -0.41900215 -0.63303035 -0.75722855 0.7817381 -0.74020356 0.572795 -0.699659 -0.7114711 0.5391187 0.5852922 -0.5037852; 0.6903843 0.4953674 -0.68038 0.5527004 0.4447352 -0.425521 0.44174457 0.62522686 0.44164163 0.6501371 -0.67075413 -0.6443054 0.6672313 -0.53363305 -0.50114083 0.7814129 0.77120036 -0.602744 0.57437885 -0.7180806 -0.44578078 0.7062593 0.5371956 -0.5853167 0.43993753 0.5392608 0.65393996 -0.6973756 0.71932703 0.58389324 -0.6238978 -0.48483744 0.64030653 -0.60995185 0.5973629 -0.6900868 0.71285796 -0.6721173 0.5183635 -0.7327441 -0.5271344 -0.43348822 0.4146556 -0.46778896 0.5691489 -0.7963111 0.6761124 0.720403 -0.5724621 -0.673053 -0.8127061 0.5504328 -0.44262484 -0.5388838 0.5453285 0.650331 -0.6776325 -0.7602716 0.41257742 0.6566366 -0.6875532 -0.6981782 -0.7811559 0.67569387 -0.5332111 -0.69318324 0.73585546 0.69761956 -0.4181079 -0.7807768 -0.69821715 -0.5791303 0.6702477 0.4727498 0.66307765 -0.6305074 -0.49243668 0.6579235 0.4525134 -0.7072522; -0.6923658 -0.68039143 0.55146587 -0.49847046 -0.44781074 0.7445958 0.8846559 -0.6903752 0.27323902 -0.58771527 0.83977675 0.7190997 -0.83738184 -0.0088402545 0.61100906 0.40991083 -0.58399695 0.32444903 -0.7318798 -0.33022106 0.51423866 -0.7793064 0.5688578 0.5294123 -0.8206208 -0.4217915 -0.7584934 0.52005535 -0.90756685 -0.39593402 0.29625097 0.31769103 -0.5846881 0.35190973 -0.6599352 0.285169 -0.8703666 0.30657452 -0.19943288 0.4742602 0.65833765 0.57448894 -0.65411174 0.31362033 -0.42397815 0.32198444 -0.858904 0.928332 0.30581844 0.53796023 0.65271807 -0.18988168 0.5678542 -0.3739077 -0.90306294 -0.78457797 0.3882279 0.7616926 -0.41165754 -0.44199672 -0.73193735 0.6061748 0.5693313 0.16001074 -0.6618759 -0.4437501 -0.8138476 -0.77867466 0.3156713 0.6326967 0.59750605 0.53941274 -0.30350628 0.77252483 0.29075354 0.4567182 0.8652892 -0.57921636 0.8883552 0.90525424; 0.5763272 0.4461211 -0.3978487 0.7255359 0.56431973 -0.38684905 -0.74236214 0.40557858 -0.72928905 0.5258462 -0.6369265 -0.51949537 0.60607594 -0.59047496 -0.5622926 -0.6068373 0.68682504 -0.5780961 0.5473035 -0.5088301 -0.5812015 0.6405757 0.2588073 -0.7022019 0.42927578 0.4719514 0.68948704 -0.43285668 0.60112125 0.7387469 -0.51015735 -0.7226688 0.68389803 -0.77636564 0.44742173 -0.7095276 0.5450361 -0.5760157 0.54701483 -0.69501305 -0.6691372 -0.50931823 0.4139306 -0.4456497 0.5758407 -0.45063877 0.7030231 -0.21549311 -0.51756847 -0.6807914 -0.4455729 0.5400325 -0.6055207 -0.7731833 0.61762863 0.6583152 -0.46597168 -0.56088257 0.5752353 0.631629 -0.6710787 -0.42272207 -0.47098267 0.283719 -1.3787309 0.6756891 0.45880356 0.26693577 -0.549972 -0.7365217 -0.5436599 -0.45019668 0.48831707 -0.56421655 -0.60005534 -0.70078754 -0.37243307 0.49308747 -0.22536273 -0.61724174; 1.0470258 0.8668794 -0.5170899 0.31082255 0.36063948 -0.7646129 0.2839037 0.72552776 0.783656 0.61621356 -0.7335377 -0.82605463 0.96039206 -0.4239081 -0.82507336 0.65975255 0.38527516 -0.32803708 0.5829873 -0.4516856 -0.25630912 0.86104816 0.8687324 0.95692414 0.74972814 0.53174144 0.6266955 -0.284091 0.9087786 0.38855085 -0.46632433 0.7571437 0.7392949 -0.40035337 0.32454494 0.91638887 0.74814886 0.94387484 0.24721336 -0.40413892 -0.46212116 -0.38435084 0.60029864 -0.47815594 0.49183264 -0.55751276 -0.15469286 0.48959658 0.8451751 -0.8575591 0.55787086 0.25035417 -0.32549617 -0.35392785 0.75187796 0.94061285 -0.2107334 -0.55964255 -0.34977695 0.37541822 -0.56896526 -0.27582425 -0.47354344 0.8535095 -0.4943778 -0.8050244 0.94944876 0.5990058 -0.26410836 -0.8233164 -0.40347522 -0.6856916 0.5204887 0.5275282 0.84367174 -0.36523905 -0.82943213 0.4308746 0.42578 -0.56837696; -0.39911133 -0.64550537 0.44055557 -0.49629205 -0.72753996 0.6984721 0.76210034 -0.54344356 -0.025585065 -0.51346457 0.6530215 0.7049293 -0.59460986 0.53678495 0.6325343 0.7467462 -0.5517341 0.6074654 -0.39448982 0.42214555 0.7521015 -0.6235079 -0.6794542 0.7138812 -0.5757518 -0.77203584 -0.73666525 0.43493313 -0.4037123 -0.4380617 0.5381988 0.67203283 -0.51131266 0.72864693 -0.78394604 0.77515405 -0.72019076 0.46450064 -0.6596934 0.754715 0.5994642 0.6600304 -0.67939186 0.5285947 -0.53589815 0.45652607 -0.5028757 -0.71764874 0.7308636 0.74299234 0.4584333 -0.77264357 0.54115593 0.6506685 -0.5547288 -0.41358626 0.48184377 0.62362564 -0.8096545 -0.47513112 0.44923112 0.5155896 0.5885207 -0.43548852 0.91160625 0.27271572 -0.41833445 -0.3385689 0.62368214 0.46463326 0.4537948 0.41525504 -0.5034107 0.26331043 -0.40687597 0.7246371 0.7360872 -0.67849326 -0.72457564 0.6035181; 0.596761 0.43390214 -0.624461 0.5087239 0.50562185 -0.7096113 -0.46554622 0.5900611 0.77254754 0.6660571 -0.43274987 -0.69936603 0.43662253 -0.6387188 -0.5675981 -0.74041 0.517406 -0.69876236 0.61442536 -0.65526223 -0.63326687 0.7306679 0.6083677 -0.56088936 0.61169004 0.86844856 0.36636752 -0.53272784 0.60757905 0.56952053 -0.76480365 -0.6322555 0.37611586 -0.7284279 0.51849955 -0.6646659 0.72420835 -0.6058773 0.6172128 -0.45766652 -0.45727912 -0.67581993 0.6131779 -0.53327453 0.66069895 -0.73763853 0.45783424 0.5082832 -0.5086683 -0.606579 -0.53267 0.77558565 -0.5602414 -0.8150425 0.46660736 0.38122308 -0.6133908 -0.35770354 0.4789311 0.71778435 -0.81275517 -0.48050746 -0.7331555 0.5127561 -0.93478197 -0.6971728 0.43407574 0.7109965 -0.51636875 -0.73224914 -0.83043295 -0.43589815 0.66298276 -1.084355 0.6597199 -0.49938732 -0.6700786 0.68423206 0.7234394 -0.6769675; 0.6622776 0.49146408 -0.56811273 0.5579048 0.52936184 -0.6945942 -0.87478596 0.6718283 0.75866 0.4543037 -0.8249725 -0.7383978 0.61950856 -0.44209978 -0.7233629 -0.2734015 0.36203554 -0.37081367 0.5392245 -0.3373316 -0.35572186 0.61182433 0.84327173 -0.3329985 0.5138016 0.49871528 0.65317994 -0.5726642 0.4684684 0.4856519 -0.66883427 -0.3937442 0.83825797 -0.70915854 0.5435915 -0.37622994 0.90094215 -0.7143818 0.54603857 -0.58117664 -0.37388405 -0.67169815 0.6740652 -0.32488436 0.47250918 -0.33721676 0.5666176 0.25947073 -0.35720882 -0.81895024 -0.8176442 0.3743032 -0.35959652 -0.32256916 0.7881041 0.60265225 -0.16944976 -0.73749036 0.54144394 0.5361819 -0.68569 -0.6125117 -0.6659817 0.66629213 -0.7828553 -0.61635494 0.5215755 0.7077114 -0.4485583 -0.5516932 -0.6107307 -0.44955128 0.3878087 -0.8283272 1.0739809 -0.5599113 -0.58763456 0.4218029 0.48027956 -0.62468433; -0.67541087 -0.5914437 0.5761122 -0.5099867 -0.7096061 0.73973167 -0.6940735 -0.60467064 -0.6457843 -0.58989054 0.64736366 0.5602039 -0.64211386 0.43895808 0.44680664 0.9122978 -0.5560715 0.5848061 -0.47659266 0.4943016 0.46250916 -0.57833844 -0.54876214 0.63871723 -0.6815348 -0.5695285 -0.42475164 0.7890623 -0.4658386 -0.49773055 0.43706563 0.46071723 -0.7714067 0.46359742 -0.43941706 0.5191171 -0.4748532 0.717535 -0.4433866 0.69798404 0.548373 0.77872515 -0.5820708 0.6386112 -0.719549 0.4818507 -0.40922034 -0.5501496 0.68294156 0.559577 0.6753365 -0.5067533 0.68588966 0.50938046 -0.6326348 -0.74235386 0.48945847 0.5744609 -0.576454 -0.70368177 0.49676585 0.5977159 0.5296727 -0.49999604 0.62880695 0.6692032 -0.44720078 -0.399139 0.65958303 0.51168865 0.626945 0.7691204 -0.673702 -0.7264146 -0.7035167 0.6972104 0.6323983 -0.7835514 -0.6591812 0.6148061; -0.5010122 -0.5428109 0.7488192 -0.63787 -0.59112227 0.71678513 -0.7162431 -0.718783 -0.38094878 -0.5486876 0.647234 0.4998782 -0.5054506 0.5472155 0.39244473 -0.3563447 -0.7583625 0.56887484 -0.44648084 0.5679527 0.41033682 -0.68160564 -0.62894464 0.57339525 -0.5228516 -0.76230174 -0.4022474 0.6284859 -0.66352594 -0.8029454 0.76974744 0.46764055 -0.73518175 0.47561604 -0.4254515 0.6310815 -0.54006666 0.6213807 -0.5664955 0.5099681 0.74546367 0.59407866 -0.4954521 0.5460067 -0.6449872 0.6939265 -0.6677011 -0.59723437 0.6900429 0.48487857 0.40195093 -0.43008938 0.52744544 0.623448 -0.41228613 -0.6675427 0.650829 0.5284497 -0.49461 -0.7793258 0.74921095 0.49205515 0.43602178 -0.551803 0.7248839 0.5797422 -0.59071857 -0.5298087 0.6933431 0.73847866 0.48894235 0.40392894 -0.5318837 -0.7240168 -0.5636952 0.5762018 0.75931805 -0.7872253 -0.71909094 0.48434365; 0.5606626 0.6981536 -0.42625147 0.6050136 0.35810426 -0.67605436 -0.42479128 0.45176116 0.86537033 0.5641815 -0.68479526 -0.5114265 0.81057054 -0.6042409 -0.7023349 0.08208024 0.70829594 -0.41853118 0.5323691 -0.57632047 -1.0728371 0.80392784 0.65337145 -0.32587457 0.70135456 0.44065228 0.38944456 -0.62249017 0.610404 0.40832844 -0.37849015 -0.4012702 0.82130116 -0.4462011 0.45117378 -0.3865786 0.6316378 -0.49001968 0.5079143 -0.407491 -0.55543524 -0.64401025 0.8004218 -0.5191046 0.44789904 -0.3721961 0.48481032 0.4406992 -0.41157672 -0.63338214 -0.704825 0.59038854 -0.48101696 -0.73577076 0.8173305 0.7186021 -0.46931016 -0.6542882 0.2971203 0.49210235 -0.7158216 -0.45112824 -0.4109017 0.85625005 -0.66433424 -0.89130676 0.6682156 0.7482407 -0.5329735 -0.82268584 -0.52045274 -0.51335037 0.58182245 0.4574065 0.5554038 -0.5398453 -0.6530535 0.52310526 0.58888334 -0.4758767; 0.72444195 0.4582245 -0.7326833 0.5643069 0.52694374 -0.7380107 -0.6961198 0.5362086 0.5332768 0.4368179 -0.66459644 -0.5487981 0.5416549 -0.4803289 -0.77901226 -0.66210806 0.71686995 -0.49983066 0.6165605 -0.56472266 -0.6010304 0.5269823 0.66708493 -0.6574615 0.56398046 0.42275274 0.69445246 -0.78637326 0.5749026 0.64632493 -0.4603962 -0.55143696 0.53659266 -0.75197357 0.6336983 -0.5232083 0.56011546 -0.55786824 0.42962354 -0.6720919 -0.49904227 -0.6995565 0.5489027 -0.48385063 0.5454786 -0.44692913 0.5814565 0.7541674 -0.5337175 -0.6331988 -0.7707659 0.45246938 -0.43932167 -0.69327325 0.7694497 0.56582755 -0.75552356 -0.5995909 0.6762811 0.41820645 -0.6343601 -0.60276866 -0.7890978 0.53225136 -0.5063513 -0.75580233 0.5006813 0.6966447 -0.436311 -0.48784208 -0.43201295 -0.65015286 0.5518998 0.45912895 0.68117094 -0.7236891 -0.7580731 0.65552056 0.67585504 -0.56967324; -0.51023275 -0.405077 0.61900514 -0.6079379 -0.47847182 0.5700619 -0.72264475 -0.61356497 -0.37857568 -0.7313128 0.47736207 0.52302015 -0.76170355 0.61281645 0.51509637 -0.543821 -0.69731724 0.5232618 -0.5569494 0.62397945 0.6665671 -0.5514779 -0.45099774 0.78208756 -0.5937153 -0.43935496 -0.5885327 0.782353 -0.55156904 -0.7709969 0.7256594 0.5665244 -0.43543854 0.6801686 -0.68128306 0.56166625 -0.6547507 0.6627338 -0.6466846 0.5072735 0.4453341 0.7613697 -0.44689095 0.65767455 -0.4271727 0.73456717 -0.46824986 -0.75328356 0.46137744 0.5343807 0.63187355 -0.69850093 0.44868657 0.4582077 -0.4745433 -0.75671405 0.7696368 0.6834913 -0.63558346 -0.6410082 0.71248025 0.55520105 0.50534683 -0.5037765 0.5995894 0.6618582 -0.5233063 -0.77256423 0.5676139 0.7863866 0.6005284 0.59949696 -0.4273978 -0.482915 -0.6468169 0.506299 0.7051566 -0.6153778 -0.47655934 0.5512728; -0.7830003 -0.73310626 0.4820952 -0.40672147 -0.7202991 0.80047333 -0.52790457 -0.5719213 -0.71066946 -0.48251504 0.652208 0.7433153 -0.8393791 0.5968836 0.72479296 -0.18746868 -0.68214375 0.7517426 -0.7429832 0.48461106 0.697142 -0.516548 -0.7665455 0.5602973 -0.510935 -0.73540896 -0.5353209 0.688775 -0.7137729 -0.64092284 0.6141462 0.68208534 -0.5124639 0.74905896 -0.5821788 0.7161036 -0.58839595 0.46587935 -0.7703525 0.6121857 0.70408213 0.683876 -0.6474243 0.49372008 -0.5317529 0.40674403 -0.50874925 -0.5141136 0.5827967 0.47471187 0.81851566 -0.6380725 0.4282465 0.62941504 -0.48264983 -0.6556091 0.48515287 0.69155866 -0.6568067 -0.7628138 0.5996781 0.62679696 0.6984752 -0.78659457 0.4392647 0.6720674 -0.64058274 -0.8164931 0.62970436 0.72390306 0.72358453 0.4914907 -0.533727 -0.46087947 -0.8044771 0.56822234 0.72419065 -0.5725044 -0.55361843 0.44200414; 0.49670222 0.49114594 -0.6963111 0.592207 0.4980356 -0.5349726 0.6994776 0.6219561 0.5820821 0.422408 -0.65084535 -0.45551115 0.544255 -0.5615192 -0.4341148 0.5375637 0.4533351 -0.495125 0.6850898 -0.56603724 -0.56830263 0.75610185 0.72955257 -0.46816787 0.6675977 0.7718619 0.5489083 -0.62347746 0.526206 0.74061716 -0.46136108 -0.6169684 0.6102796 -0.57512265 0.6056216 -0.58439773 0.6620709 -0.5493618 0.5560958 -0.79436606 -0.5637157 -0.79064256 0.65317184 -0.74196523 0.5039766 -0.5211241 0.5131176 0.42114064 -0.5077395 -0.5398513 -0.6787155 0.4775788 -0.76079726 -0.66116357 0.685138 0.5978395 -0.51102877 -0.49205658 0.54943043 0.47249734 -0.6380398 -0.5086323 -0.4592323 0.6056684 -0.61975724 -0.42969146 0.4871622 0.6978162 -0.5123488 -0.53328943 -0.5168815 -0.40562975 0.4991979 0.5913845 0.68980736 -0.61001486 -0.73569924 0.43175232 0.7021749 -0.59163934; -0.6454732 -0.4024187 0.5124096 -0.46691093 -0.70976114 0.61017346 0.7898896 -0.51042944 0.63992435 -0.76457804 0.49291795 0.73590046 -0.73079264 0.43568134 0.73934585 0.5827897 -0.76312 0.4070936 -0.41954938 0.5916974 0.58480716 -0.7723128 -0.37944248 0.41915017 -0.71240836 -0.4361831 -0.57873607 0.43955362 -0.61978334 -0.5374359 0.760192 0.5447141 -0.58853143 0.73949945 -0.69942653 0.7273109 -0.6435661 0.6099241 -0.40316415 0.6013844 0.47107092 0.61614513 -0.5776632 0.7073133 -0.63821113 0.71243066 -0.4938591 -0.2489395 0.7737425 0.56819886 0.7676542 -0.66085505 0.7651681 0.7744058 -0.7279134 -0.6653474 0.79684436 0.59983206 -0.4718053 -0.61830485 0.5066911 0.6154132 0.4476999 -0.38813224 0.9178315 -0.64012027 -0.54767203 -0.45862237 0.61678606 0.53231126 0.75837713 0.62512743 -0.6991457 0.46836478 0.72661877 0.51004773 0.42946213 -0.64638 -0.70181865 0.41941628; -0.44318298 -0.68875986 0.5168043 -0.5526032 -0.43304217 0.4910622 -0.43858206 -0.7798409 -0.36186203 -0.42172027 0.583827 0.73306465 -0.7753272 0.59104115 0.5640906 0.11927822 -0.45980504 0.4528206 -0.41471976 0.6125779 0.4127044 -0.44697702 -0.6054483 0.5001761 -0.45183983 -0.50665987 -0.49884284 0.54491997 -0.42868003 -0.6767293 0.52187157 0.7168908 -0.71140885 0.405449 -0.4395439 0.47685054 -0.41003615 0.7408283 -0.5710204 0.6546006 0.6275396 0.76646113 -0.6925205 0.79249936 -0.4345434 0.60462385 -0.49356315 -0.5181935 0.47320563 0.5463243 0.6445621 -0.6899103 0.45191944 0.51318514 -0.6397761 -0.53083056 0.7825924 0.4753459 -0.6548987 -0.41026488 0.7309062 0.4550119 0.7904653 -0.67992663 0.65645343 0.58009964 -0.4403952 -0.6391608 0.5334121 0.65346146 0.5479783 0.5905126 -0.69554895 -0.4735042 -0.6871987 0.76243514 0.5900523 -0.6214571 -0.4807129 0.46288532; -0.6854915 -0.60690135 0.5543723 -0.7337229 -0.47590917 0.52965736 -0.7076711 -0.74291486 -0.41012013 -0.57981205 0.62083596 0.5124888 -0.5906427 0.54043216 0.4469362 -0.56211054 -0.5050269 0.48279378 -0.61057895 0.5590875 0.44058755 -0.50309116 -0.5728043 -0.4336845 -0.5641334 -0.65263534 -0.49577156 0.46206653 -0.4188252 -0.5971518 0.43866542 0.7409765 -0.77382094 0.6085247 -0.7609022 0.64822286 -0.47318587 0.55327964 -0.5556969 0.5613718 0.5158969 0.6237709 -0.71563095 0.76155406 -0.49720135 0.56719327 -0.531126 -0.7043119 -0.6445073 0.41108212 -0.8106725 -0.6871147 0.5174539 0.6129659 -0.5545048 -0.40786475 0.4962075 0.3932343 -1.2331605 -0.7329864 0.60163754 0.57791597 0.49148902 -0.4917986 0.5323959 0.5864553 -0.73930115 -0.45108798 0.50663847 0.7775139 0.64489526 0.49081197 -0.47521478 -0.61184376 -0.41144696 0.6213409 0.5717648 -0.5133803 -0.68291396 0.46566698; -0.6463361 -0.70975834 0.69944125 -0.7817753 -0.696821 0.7013465 0.57901806 -0.6971079 0.28900826 -0.40277618 0.60116315 0.47789708 -0.6925413 0.78287524 0.42863977 0.5650684 -0.68132484 0.59768057 -0.4098413 0.70022285 0.66997695 -0.38125625 -0.5013352 0.6430466 -0.44116524 -0.6563199 -0.7033985 0.83079064 -0.6339422 -0.7268631 0.59948534 0.48948532 -0.7098123 0.8574552 -0.5226201 0.6571629 -0.45927608 0.64548415 -0.5102074 0.78174084 0.71206164 0.5750658 -0.5711631 0.49362364 -0.7234799 0.82284737 -0.3723225 0.97464335 0.5812307 0.4596224 0.42891422 -0.4547417 0.6136185 0.6274475 -0.38507196 -0.48317823 0.43853357 0.36694118 -0.6620214 -0.78156626 0.45800847 0.54088026 0.678265 -0.70444596 0.08860962 -0.77210414 -0.42644677 -0.58804905 0.78594774 0.6584726 0.69266236 0.7310466 -0.72494465 0.6287104 0.36402646 0.6362671 0.67665577 -0.5843512 0.77056307 0.5115955; 0.22062926 0.25371483 0.59305865 -0.46906695 -0.43281594 0.7299081 -0.4387373 0.62074965 -0.6428825 -0.77047116 0.9976969 -0.57143885 0.22621165 0.4317596 0.686381 -0.6637925 -0.04581766 0.21508893 0.3045839 0.6003751 0.3891363 0.51253843 -0.9041358 -0.59299135 -0.09771135 -0.5693825 0.5202659 -0.89229065 -0.8127906 0.90776235 -0.8524181 -0.5391377 0.2972612 0.23646018 -0.3694955 -0.7535449 -0.83906406 -0.74305797 -0.71533173 -0.99554586 0.57462066 0.59788954 0.25828397 0.21560483 -0.6138112 -0.91551363 0.5607238 -0.3715929 -0.6528754 -0.27671367 -0.5596007 -0.51052684 -0.24627636 0.45888633 0.63566524 -0.8558264 -1.3087901 -0.68859285 0.5283095 -0.4801174 0.39990592 0.18382743 -0.9506712 -0.91178524 0.36356106 0.5719815 -0.74830115 -0.8274098 -0.9001226 -0.5839017 0.5141702 0.28066045 0.9135218 -0.5437354 -0.87852335 -0.8574334 0.7803233 -0.6277476 -0.5001995 0.0409377; -0.4024484 -0.43433815 0.46044725 -0.5637304 -0.44708 0.6106667 -0.5941332 -0.575451 -0.62617123 -0.40084168 0.4211874 0.61508787 -0.5174069 0.5687799 0.38253072 -0.35793644 -0.64665806 0.5180193 -0.55424833 0.57385206 0.584888 -0.36231536 -0.7055208 -0.46285883 -0.62767756 -0.53882134 -0.7581073 0.47640663 -0.5727611 -0.7595138 0.73781365 0.74745 -0.3932576 0.5162723 -0.7745711 0.81520194 -0.75201225 0.6012621 -0.5852282 0.7256518 0.70920897 0.50576925 -0.46963802 0.55847543 -0.5561132 0.5649233 -0.3125386 -0.681691 -0.31659693 0.6849942 -0.8882579 -0.77510947 0.47787726 0.7458484 -0.47324738 -0.6418412 0.83324635 0.5289833 -0.015253378 -0.60473645 0.7476808 0.7209276 0.46055338 -0.5223372 0.70181286 0.6082037 -0.39252898 -0.6806538 0.6906396 0.42824066 0.787364 0.5417544 -0.7085188 -0.46352357 -0.678351 0.56782395 0.6168651 -0.79482573 -0.56044126 0.61461896; -0.572101 -0.6282563 0.7069469 -0.5737264 -0.5427103 0.47552037 0.87784415 -0.47997704 -0.76879627 -0.48467147 0.55637544 0.64955765 -0.70155114 0.4728189 0.75163853 0.5135757 -0.59785825 0.6874397 -0.4101132 0.4448963 0.8102332 -0.55943644 -0.57435435 0.3972726 -0.72869074 -0.7524292 -0.7006921 0.55672395 -0.73368675 -0.68600756 0.6761881 0.45681742 -0.43182185 0.6700703 -0.42513368 0.69720954 -0.6784525 0.5387788 -0.45496777 0.740292 0.4135024 0.4206422 -0.6913278 0.5200205 -0.58446634 0.74267465 -0.6052979 -0.40432546 0.7347985 0.6900578 0.6120168 -0.46135113 0.640706 0.572648 -0.5906418 -0.65254563 0.46578825 0.5177605 -0.4367365 -0.44544917 0.6361817 0.5933556 0.40451723 -0.78692836 0.50380415 0.61125344 -0.67090666 -0.42703646 0.53013587 0.6660144 0.73437357 0.7699517 -0.76937795 -0.62593484 -0.58889437 0.5469526 0.624288 -0.64604926 -0.5311983 0.5398324; -0.533972 -0.4490722 -0.48578992 0.63873947 0.55073917 -0.6038344 0.63723314 -0.55019027 0.66466695 0.9080596 -0.9051717 0.38278544 -0.49428198 -0.5968945 -0.6581403 0.8310781 0.13644536 -0.44181484 -0.3471817 -0.6154009 -0.6499888 -0.42549652 0.8243053 0.5430309 -0.13211817 0.55125105 -0.6202411 0.777166 0.8708791 -0.57705104 0.60139143 0.4977911 -0.574588 -0.7131148 0.5131089 0.7817233 0.8690532 0.6430013 0.43555441 0.64502466 -0.528008 -0.63386697 -0.36526623 -0.19818698 0.7604659 0.6065691 -0.5450274 0.36908144 0.85370684 0.46482506 0.7366122 0.5661455 -0.06876656 -0.54366213 -0.64904624 0.8255271 0.6406336 0.49395192 -0.71050143 0.46772006 -0.50821316 -0.22542405 0.7120974 0.84688354 -0.31165686 -0.7161481 0.6527253 0.5376726 0.5691874 0.3976258 -0.4353548 0.284532 -0.8213055 0.6851242 0.7848838 0.21461403 -0.7104427 0.40034747 0.5790893 0.20479743; 0.5232919 0.71308595 -0.5155606 0.4390751 0.7086023 -0.5537415 -0.6279631 0.6541278 0.57471967 0.78487253 -0.72708344 -0.5086049 0.6198133 -0.7712123 -0.43685302 -0.5196094 0.5126169 -0.46884334 0.5778567 -0.43939313 -0.7290943 0.44029972 0.4908539 -0.79756683 0.42555696 0.71618414 0.7634805 -0.69637406 0.63349557 0.4393798 -0.7003277 -0.4473282 0.6644342 -0.4426687 0.46312472 -0.738956 0.6369203 -0.7939522 0.5754162 -0.57852536 -0.46041423 -0.6425074 0.5992671 -0.66865885 0.6371653 -0.4968074 0.69453204 0.44697377 -0.7848458 -0.600857 -0.52975154 0.42482406 -0.5154686 -0.43249863 0.58427244 0.67571795 -0.7285415 -0.7039383 0.75000733 0.4254054 -0.43796763 -0.42002624 -0.48030913 0.612965 -0.57942665 -0.5538959 0.4001494 0.49587065 -0.66961443 -0.7493857 -0.7599463 -0.70250195 0.68966067 0.6320073 0.464802 -0.6184034 -0.6984886 0.51846004 0.7692319 -0.5023299; 0.64406615 0.4305593 -0.5123122 0.6132447 0.6443844 -0.7196739 0.83977896 0.40543813 0.38195238 0.7292234 -0.6838268 -0.6079654 0.54136586 -0.73062783 -0.5677809 0.49820176 0.4260539 -0.6059202 0.5642101 -0.77281314 -0.4538392 0.7219752 0.59739155 -0.3069297 0.42499554 0.74129045 0.72605824 -0.68768233 0.47624654 0.7440669 -0.57332015 -0.83312154 0.52442354 -0.49156561 0.5385464 -0.71630466 0.6863057 -0.67441505 0.53555226 -0.57884544 -0.6696296 -0.77853286 0.7280566 -0.7124423 0.7021241 -0.7884632 0.55965453 0.55758035 -0.013069293 -0.4014426 -0.14871389 0.69653213 -0.50577295 -0.6242481 0.64858174 0.78027403 -0.5974435 -0.6763053 0.7628068 0.7842195 -0.5771572 -0.44184545 -0.5702074 0.68243575 -0.54855776 -0.6087679 0.4371597 0.374484 -0.7577686 -0.4626044 -0.42733955 -0.6213793 0.44636732 0.7524642 0.45533958 -0.53400403 -0.4220199 0.68879616 0.55696225 -0.44997647; 0.5734591 0.6371391 -0.5204834 0.67977035 0.80650675 0.6385015 -0.49682426 0.6570278 -0.68870735 0.7671951 -0.5174453 -0.49467084 0.44879517 0.46392962 -0.060180232 -0.75402933 0.7147242 -0.5917421 0.57182974 0.4864726 -0.8356797 0.5707524 -0.90346813 -0.7297953 0.6579757 0.5042343 0.76218367 -0.85396683 0.47573486 0.67114407 -0.59825003 -0.7724988 0.60854864 -0.6231552 0.59297913 -0.51769674 -0.34592798 -0.62772685 -0.7087837 -0.8267301 -0.8220004 -0.8215928 0.42118943 -0.43984842 0.5393675 -0.6973929 0.6174791 -0.59703654 -0.778182 -0.5896368 -0.44510943 -0.55089647 -0.80781615 0.66755897 0.55727243 0.68312454 -0.3159047 -0.7331029 0.41575024 -0.17905195 0.71848214 -0.78141785 -0.5977601 -0.7036944 0.8726585 0.67240834 0.34186724 -0.0440988 -0.5360509 -0.73024344 -0.50740355 -0.51689905 0.6232332 -0.46176678 -0.63954777 -0.32856685 -0.6875098 0.8301388 -0.73262143 -0.6661827; -0.5173374 -0.5078473 0.773012 -0.7959942 -0.7863219 0.6390312 -0.5113926 -0.49137226 -0.4703856 -0.6721618 0.58549726 0.56717485 -0.5696015 0.6148109 0.496987 -0.61873865 -0.56906265 0.79310364 -0.5064654 0.4931217 0.590668 -0.5110988 -0.5290033 0.70535105 -0.47524646 -0.7249266 -0.5982781 0.7781313 -0.64848584 -0.6739458 0.6884441 0.7700523 -0.6016872 0.5655991 -0.555516 0.70309246 -0.5279499 0.4841927 -0.46329597 0.5592389 0.45205185 0.629732 -0.44520727 0.41666505 -0.42672637 0.6389565 -0.50160474 -0.5721287 0.5755983 0.63386047 0.71286577 -0.56229347 0.47117734 0.42571214 -0.6319138 -0.55846304 0.75179064 0.51440114 -0.7251954 -0.43130502 0.47333202 0.465002 0.49549928 -0.66652995 0.7182534 0.53413695 -0.51211774 -0.44382158 0.6155371 0.5854091 0.63527715 0.49897274 -0.56406444 -0.4823685 -0.49987072 0.6393195 0.6185537 -0.48059642 -0.7650008 0.76177394; -0.5345969 -0.79819095 0.63945436 -0.61275434 -0.6188298 0.3860318 -0.47669533 -0.83258444 -0.59839183 -0.6254077 0.5187244 0.5306625 -0.48806491 0.50411326 0.61624235 -0.41464838 -0.7796846 0.785875 -0.7238786 0.4659928 0.46578133 -0.5415056 -0.62346894 0.46172327 -0.47369725 -0.49308988 -0.6618674 0.80401933 -0.4809602 -0.67272675 0.75059664 0.47974363 -0.49494308 0.76880234 -0.52842057 0.50510687 -0.4397646 0.68345976 -0.73000777 0.769043 0.8051742 0.6082702 -0.72538865 0.780779 -0.5484747 0.5231281 -0.6310699 -0.4208025 0.395152 0.65322 0.750655 -0.5739571 0.71130335 0.46105626 -0.53700536 -0.63852876 0.60536593 0.65867245 -0.38896018 -0.62300414 0.716815 0.7172548 0.81584036 -0.72279984 0.2628916 0.45879683 -0.37581056 -0.7068545 0.3716284 0.60466504 0.6631763 0.58022964 -0.71438205 -0.6081777 -0.46380427 0.6438947 0.6598219 -0.5645841 -0.75717556 0.42354828; -0.21855332 -0.603724 0.47017306 -0.58901757 -0.7026565 0.5608577 -0.5724887 0.6977399 -0.74807805 -0.42037246 0.8246056 0.4597358 -0.688909 0.48956513 0.84519005 -0.6958255 -0.41562924 0.4442135 -0.5468105 0.5955389 0.5656886 -0.3350992 -0.65304565 -0.6178862 -0.5241937 -0.49983832 -0.31666008 -0.5533949 -0.71103156 -0.42059952 -0.8622817 -0.70373815 -0.7251771 0.3202689 -0.45029688 -0.56893337 -0.90139085 -0.8799904 -0.32463452 0.35404164 0.6110592 0.6459355 -0.8176259 0.6183246 -0.47744468 0.5372839 0.56995666 -0.5843616 -0.80367196 -0.014609926 -0.37646258 -0.4075635 0.389818 0.5325982 0.29108503 -0.7805108 0.18674608 -0.39377952 0.4222093 -0.41237432 0.6900893 0.5062674 0.35885346 -0.79644316 0.39030865 0.6343672 -0.87726253 -0.60374826 -0.21725182 0.8165064 0.35461375 0.7432436 0.3361991 -0.5144158 -0.74342877 0.29273093 0.6587315 -0.32070613 -0.35274902 0.62907994; 0.44787085 0.5930318 -0.2671775 -0.34476626 -0.6624121 0.4041754 -0.47786638 0.8195169 -0.66038686 0.39060614 0.3305715 -0.65941405 0.67477566 0.59304047 0.64055395 -0.5559631 0.27596098 -0.53244454 0.6760158 0.53193134 0.3117099 0.616468 -0.5324807 -0.734161 0.69966984 0.015959471 0.71885073 -0.5175587 0.7916006 0.44094574 -0.684213 -0.42656446 0.74708784 -0.4961347 -0.77008134 -0.70166934 -0.4828578 -0.4044204 -0.6553997 -0.7022202 0.36561278 0.69001836 0.49487877 0.78893906 -0.76799464 -0.36157754 0.80559057 -0.6419464 -0.46952474 -0.54111284 -0.60891765 -0.53766423 -0.43550032 0.7888128 0.66892594 0.7357496 -0.13537323 -0.78680724 0.71804965 -0.45448843 0.6259154 -0.48795488 -0.44610336 -0.674479 0.6718422 0.71219826 -0.17098396 -0.42132062 -0.39374626 -0.7025216 0.67506546 -0.7175773 0.4971037 -0.7278151 -0.6236721 0.029518329 -0.02886901 -0.25638235 -0.8086403 -0.76558894; 0.9605602 0.6304172 -0.3812783 0.6600672 0.38437 -0.76538587 0.51885384 0.8885758 0.9496666 0.62682796 -0.662842 -0.49910238 0.9305439 -0.48987636 -0.897334 0.68199927 0.35786974 -0.4811532 0.76714355 -0.487068 -0.31080204 0.60843784 0.6608583 1.0126826 0.799317 0.56123644 0.5527537 -0.395612 0.8616205 0.4077788 -0.4429892 1.1312302 0.6372805 -0.31878984 0.65996295 1.0471622 0.62904644 0.7252274 0.5738648 -0.2948104 -0.6558493 -0.49444178 0.8966648 -0.30849674 0.37265542 -0.552926 0.99619275 0.47688904 1.0075793 -0.7889193 0.5224019 0.37025002 -0.39480528 -0.3332141 0.62103105 0.5474575 -0.34861597 -0.8554775 -0.23617585 0.6668067 -0.6673708 -0.33463335 -0.4001102 0.6185938 -0.4515636 -0.7267892 0.65543157 0.6669688 -0.24794945 -0.6486582 -0.3109344 -0.5227409 0.24134652 0.59855694 0.9070958 -0.24299082 -0.7526655 0.43350208 0.38089395 -0.85552067; -0.82115746 -0.80531377 0.4001925 -0.39450076 -0.35555646 0.7964382 -0.52680856 0.6245092 -0.56676054 -0.60460633 0.7667337 0.7538163 -0.641047 0.582927 0.806071 -0.641386 -0.6159489 0.4093102 -0.5011562 0.38260293 0.53795123 -0.60653067 -0.7076581 -0.8014046 -0.8477537 -0.6587308 -0.52774334 -0.7699934 -0.6647416 -0.3072222 0.17882216 -0.59340423 -0.76581675 0.52649665 -0.7062563 -0.78616625 -0.71766514 -0.7949331 -0.7260413 0.33121136 0.67882633 0.6094857 -0.62010235 0.48432288 -0.60366255 0.5547895 0.63784426 -0.69955844 -0.8802742 0.55896693 -0.38302562 -0.7271917 0.39114705 0.69456553 -0.49287847 -0.81384075 0.23316494 0.45850185 0.465694 -0.5835317 0.48518243 0.536079 0.3884094 -0.64759004 0.45224276 0.787326 -0.88373196 -0.68412095 0.26207578 0.666087 0.3687604 0.59231186 -0.52822 -0.5812711 -0.57823646 0.52801937 0.8393663 -0.6606016 -0.3468236 0.62547994; -0.3989995 -0.63591367 0.48411104 -0.76085037 -0.5702561 0.72888654 -0.13566528 -0.7643332 -0.6893476 -0.6094528 0.563113 0.46513858 -0.495175 0.67049026 0.6016682 0.35505998 -0.7668818 0.71897197 -0.5014723 0.7648717 0.656285 -0.6577875 -0.59480524 0.39843768 -0.48835543 -0.7903796 -0.62356967 0.65350693 -0.6770944 -0.6392448 0.4832137 0.46728665 -0.7681538 0.50248075 -0.64451444 0.68549025 -0.56421924 0.53284633 -0.69263667 0.65087295 0.4540336 0.5718926 -0.74732655 0.6842891 -0.4333508 0.808288 -0.47691295 -0.38772961 0.6847106 0.70651925 0.7162186 -0.54552263 0.5228799 0.5110768 -0.48174325 -0.4898739 0.34823373 0.62389857 -0.4671147 -0.5563975 0.52464795 0.6930213 0.48206905 -0.6304276 0.19157474 0.46052387 -0.60365564 -0.64894205 0.47423333 0.696212 0.7178567 0.43204856 -0.55111223 -0.59884286 -0.5035665 0.68444335 0.6767556 -0.78002715 -0.4503925 0.7133965; 0.6111997 0.34829378 -0.8412637 0.5935401 0.64020574 -0.6281883 0.5994958 0.028046379 0.27818745 0.66765356 -0.67796534 -0.7169445 0.53984374 -0.60151225 -0.5143121 0.46330544 0.82390887 -0.7877161 0.41959807 -0.8733378 -0.639539 0.5184516 0.5966415 0.24394745 0.31766218 0.85859376 0.6215376 -0.47897616 0.4589722 0.7158447 -0.55618674 0.51360154 0.3992528 -0.75948095 0.80189115 0.23563199 0.6499164 0.31586695 0.62466323 -0.83745134 -0.7552456 -0.65546685 0.43528542 -0.75756973 0.8562154 -0.8126807 -0.8468218 0.63794404 0.25382432 -0.55649734 0.93110335 0.88030964 -0.599501 -0.7560403 0.28295767 0.5537614 -0.53504914 -0.5035347 -0.37992087 0.7248037 -0.56902915 -0.86913806 -0.51376617 0.31246188 -0.43189055 -0.33826062 0.38833407 0.55038863 -0.66803455 -0.45130023 -0.77014446 -0.3993509 0.71269965 0.5580093 0.417387 -0.7864751 -0.60990334 0.8680403 0.6413525 -0.64509207; -0.4885293 -0.58326286 -0.56901914 0.520052 0.51204056 -0.74305516 0.52238774 -0.7947391 0.4342578 0.69842964 -0.5958787 0.6197661 -0.7399901 -0.47387925 -0.7493766 0.61420214 0.7903309 -0.58256865 -0.43883875 -0.74791354 -0.80564183 -0.6589238 0.4811396 0.73932683 -0.3643623 0.78377587 -0.6658159 0.7035551 0.7380969 -0.46458563 0.5953429 0.50206834 -0.42918727 -0.7961391 0.7630496 0.4304087 0.4329504 0.6310595 0.5481934 0.3172393 -0.6383495 -0.80779016 -0.35812506 -0.64928824 0.4896153 0.40690213 -0.6661329 0.52951103 0.7548158 0.5148508 0.69490266 0.78917944 -0.8022892 -0.83957887 -0.6870475 0.51436216 0.2912223 0.68088347 -0.45670882 0.69000065 -0.72552115 -0.7987526 0.34719437 0.7432626 -0.20555112 -0.5862634 0.4950996 0.5080475 0.61175 0.4901126 -0.6028929 -0.35647604 -0.4649708 0.68941647 0.3861412 -0.4231586 -0.478427 0.64967203 0.4808686 -0.120669104; -0.71478456 -0.46687886 0.410028 -0.78098506 -0.6954813 0.422661 0.38385737 -0.41308546 -0.45292008 -0.4506991 0.5768737 0.6460141 -0.3673757 0.4115776 0.41556075 0.79352623 -0.51685715 0.5297876 -0.519417 0.4722979 0.41384807 -0.5550201 -0.6427206 0.78632283 -0.6844653 -0.52172875 -0.48078525 0.5195784 -0.7485677 -0.68751025 0.46287945 0.7088323 -0.72144943 0.49517506 -0.5549641 0.8064979 -0.46346006 0.79167217 -0.7835249 0.57636875 0.43642193 0.45748836 -0.51336557 0.5888203 -0.5776667 0.7513063 -0.3927323 -0.6022529 0.6401311 0.42125934 0.5287386 -0.65156084 0.53543144 0.49551538 -0.414077 -0.44746107 0.551619 0.7062477 -0.5071506 -0.5243136 0.5172771 0.5188406 0.7782303 -0.6438804 0.92669654 0.4758292 -0.7177658 -0.6094186 0.538198 0.56485844 0.6260101 0.5244465 -0.5224655 -0.43975422 -0.3800932 0.61366487 0.39230826 -0.66182214 -0.5150524 0.5369375; 0.59705365 0.63196665 -0.45924616 0.73070234 0.6079291 -0.5188402 0.45607558 0.54297227 0.39125133 0.5114235 -0.5259666 -0.6494191 0.4699865 -0.760728 -0.68074876 0.6259233 0.4956791 -0.8193286 0.5710036 -0.50132066 -0.5697471 0.5361286 0.4657442 0.013149617 0.60621995 0.6147225 0.8007897 -0.4746506 0.45250916 0.50015056 -0.7966004 -0.70393693 0.5282069 -0.70201814 0.5177519 -0.55861443 0.62131256 -0.40696472 0.81019235 -0.73360497 -0.71051556 -0.69588137 0.5391329 -0.6484278 0.44608232 -0.5580733 0.6158401 0.5017803 0.39256197 -0.4512791 -0.49476853 0.52207726 -0.7612488 -0.8000496 0.5321616 0.51826775 -0.6328568 -0.6981153 0.47594696 0.75089586 -0.56077874 -0.7984075 -0.7907657 0.42708334 -0.35119456 -0.5742499 0.45790958 0.5930253 -0.4373765 -0.6661536 -0.4760256 -0.69540757 0.8054203 0.621501 0.596865 -0.47435933 -0.62820804 0.5621235 0.5039426 -0.43588582; -0.44941178 -0.6459833 0.7769393 -0.54207844 -0.70650136 0.7657965 0.36968857 -0.7873351 -0.61539894 -0.70604044 0.7560397 0.7831155 -0.56967175 0.63253605 0.7632412 0.32077262 -0.5772356 0.78841954 -0.51613855 0.661998 0.41225514 -0.5818133 -0.59682107 0.6615298 -0.7134705 -0.7179481 -0.51079047 0.47599077 -0.5055323 -0.5259082 0.5466474 0.68263096 -0.57665956 0.5079244 -0.46472684 0.557282 -0.70830435 0.66852295 -0.7251586 0.5162447 0.4256422 0.73396623 -0.65572757 0.5313568 -0.5779893 0.5695859 -0.7771458 -0.6340742 0.7059152 0.73152083 0.6085521 -0.7063194 0.5276878 0.4927768 -0.723979 -0.7503154 0.47682333 0.5411078 -0.45240468 -0.5742798 0.63017756 0.7626513 0.5269501 -0.53078884 0.44847098 0.54996777 -0.7640968 -0.6496616 0.7420393 0.49759766 0.4793649 0.6596798 -0.79251415 -0.48420718 -0.60031146 0.65474135 0.6838003 -0.5197161 -0.41357034 0.51667714; -0.72077334 -0.7418019 0.79152685 -0.50989544 -0.47164837 0.66868854 -0.5593284 -0.67755234 -0.67205036 -0.48162323 0.5195058 0.6687051 -0.42268547 0.61848545 0.66721416 -0.4538461 -0.4880282 0.41628137 -0.56856424 0.6987955 0.7119883 -0.48175964 -0.42100632 0.83734524 -0.7062936 -0.46141532 -0.5920184 0.59906125 -0.6458173 -0.7953893 0.5975098 0.7381533 -0.48397672 0.5209973 -0.46297517 0.52636576 -0.43917006 0.740987 -0.5247003 0.6999472 0.6065428 0.5464971 -0.3972321 0.6363928 -0.4545863 0.43362728 -0.50315577 -0.7641962 0.8282474 0.73161453 0.5702267 -0.7068221 0.51405156 0.44411895 -0.5235747 -0.5092144 0.69915974 0.7167619 -0.82297957 -0.6951055 0.75592524 0.43713152 0.6244128 -0.40071192 0.53206533 0.55626243 -0.63658917 -0.48736677 0.506019 0.40694943 0.61259776 0.5511779 -0.69342285 -0.51282746 -0.5785066 0.58428556 0.7700165 -0.6653046 -0.55983794 0.57446206; 0.7637278 0.65365237 -0.66680884 0.51395553 0.5016826 -0.48248142 0.8263524 0.6770526 0.7270036 0.5087118 -0.6392796 -0.46612453 0.7580122 -0.64732254 -0.49760124 -0.4358714 0.6777777 -0.42686313 0.79264003 -0.54669356 -0.6765315 0.61957264 0.47520873 -0.39702964 0.6989675 0.56267357 0.68514895 -0.49499497 0.5900282 0.50647455 -0.54240674 -0.42317748 0.71984464 -0.6769332 0.43116608 -0.4036135 0.78913665 -0.41525486 0.63475585 -0.68599224 -0.47064087 -0.43567777 0.7419703 -0.76668584 0.50458467 -0.6301543 0.6022949 0.41353133 -0.49694797 -0.45339206 -0.5963494 0.5797412 -0.4117212 -0.53361124 0.7985275 0.69201326 -0.47130907 -0.7667944 0.7763023 0.61605036 -0.66933537 -0.49059743 -0.5674244 0.62448967 -0.73323816 -0.55837214 0.7692773 0.53898686 -0.51251304 -0.7047458 -0.6288325 -0.62734336 0.43365002 0.46655518 0.6404071 -0.61895126 -0.4546416 0.52643317 0.7221099 -0.66433483; 0.6630487 0.7510625 -0.6837828 0.41990107 0.75990456 -0.64503735 -0.6576765 0.39764285 0.69804066 0.61341757 -0.48582953 -0.5698454 0.55737 -0.68950874 -0.7103334 -0.51750237 0.59540254 -0.47283956 0.40874612 -0.4492384 -0.70118815 0.67913216 0.66403484 -0.56919104 0.5308884 0.5175717 0.6627201 -0.6067086 0.79125935 0.41615853 -0.4954559 -0.69301474 0.5040307 -0.4783305 0.703906 -0.75735605 0.63028616 -0.70240533 0.46281764 -0.6731465 -0.532345 -0.6872468 0.61790967 -0.4217869 0.6778933 -0.4418862 0.74820477 0.55697566 -0.63430285 -0.5239363 -0.646018 0.43091348 -0.6406773 -0.42051515 0.47257867 0.6640728 -0.56271815 -0.5920244 0.54536873 0.65030617 -0.5951088 -0.38925907 -0.7289503 0.7113974 -0.57694525 -0.6353456 0.7999587 0.72644967 -0.4877601 -0.49803215 -0.4862038 -0.4263566 0.41576996 -0.2516678 0.42712605 -0.43153566 -0.5409738 0.6316852 0.50219 -0.51505786; -0.50355893 -0.49316892 0.49251914 -0.4435023 -0.67912984 0.6445952 0.5630129 -0.7747401 -0.7547393 -0.5709817 0.65929604 0.53349805 -0.7531947 0.756341 0.47400007 0.7906458 -0.61666673 0.51369095 -0.55116135 0.71346813 0.72568625 -0.6752138 -0.64361274 0.54032403 -0.50395 -0.5559153 -0.45283455 0.5258597 -0.7006968 -0.6668301 0.6575223 0.43244985 -0.45296866 0.6130288 -0.4571991 0.56749135 -0.51241064 0.7787807 -0.47027543 0.6211492 0.43648887 0.42981642 -0.7810958 0.6202703 -0.60738647 0.41790694 -0.65768147 -0.49732062 0.7141291 0.66345567 0.42886573 -0.48017612 0.7677925 0.7276606 -0.73146355 -0.66964936 0.44572902 0.52659285 -0.76031095 -0.4598142 0.5218835 0.57305443 0.51155454 -0.56096995 0.63638717 0.67482215 -0.52446234 -0.5783194 0.6893996 0.7647914 0.4535558 0.7248859 -0.64858663 -0.745162 -0.55829775 0.48965636 0.6227954 -0.7121509 -0.6608435 0.749863; -0.53896886 -0.5796312 0.41063055 -0.5122588 -0.6697569 0.62212723 -0.7456361 -0.41535673 -0.4229516 -0.6593957 0.54916507 0.4158768 -0.633748 0.41457567 0.6474827 0.8305573 -0.4273957 0.557693 -0.7649156 0.4706299 0.6247945 -0.52142763 -0.6247582 0.4539217 -0.70549333 -0.4194003 -0.50820917 0.45532167 -0.6768758 -0.77839136 0.5370124 0.52932686 -0.6807079 0.78855354 -0.48901087 0.4333906 -0.7278119 0.44555604 -0.76418716 0.7619225 0.774339 0.548145 -0.69455826 0.73557717 -0.590834 0.48967844 -0.62818754 -0.50837463 0.6050124 0.46120048 0.76654404 -0.61784077 0.64448035 0.46467778 -0.7742182 -0.55883116 0.7053591 0.7286821 -0.46091357 -0.540838 0.45740238 0.49338365 0.55689234 -0.56219 0.56348664 0.57866716 -0.75490236 -0.69923717 0.46043777 0.7183643 0.43383864 0.49815634 -0.71688336 -0.65839565 -0.5251264 0.43240273 0.40641776 -0.42446992 -0.45507985 0.68455356; 0.7890129 0.24278696 0.5106596 -0.4933964 -0.7049899 0.46094218 -0.7016217 0.49164933 -0.41522 -0.65838355 0.3642981 0.5548991 0.80838335 0.6873851 0.36525527 -0.5221785 -0.46942875 0.6498633 -0.28983352 0.7645944 0.53141546 1.0024935 -0.37456965 -0.38580543 -0.27132732 -0.6288192 0.15222453 -0.70060927 -0.39494136 -0.86923194 -0.42960322 -0.640614 -0.39603466 0.6492362 -0.6733937 -0.6070572 -0.6253431 -0.35704938 -0.4389195 0.7768106 0.8062502 0.6696137 -0.5821042 0.7693263 -0.49337256 0.6399235 0.4738696 -0.6531432 -0.5792967 -0.68654704 -0.82233363 -0.7462623 0.74289197 0.54499906 0.7419409 -0.37980467 0.2132352 -0.9249801 0.5088468 -0.7321019 0.7980993 0.4834329 0.5408741 -0.5060908 0.6507198 0.3952069 -0.67611027 -0.47262448 -0.0070000775 0.4892355 0.5155129 0.45340246 0.51824516 -0.56626433 -0.4945155 0.7375714 0.62321347 -0.6877607 -0.7031686 0.45837122; 0.86177444 0.5649166 -0.40868357 0.39604336 0.625117 -0.69393265 0.53127253 0.5941573 0.86282945 0.38511387 -0.79465735 -0.4452251 0.75750655 -0.63622785 -0.53438455 0.78798926 0.5611812 -0.514548 0.6666498 -0.56098735 -0.447858 0.49963397 0.8258697 0.72687125 0.570566 0.45747912 0.5893976 -0.37439692 0.57735157 0.5908298 -0.54752094 -0.5159277 0.56257534 -0.4498677 0.50963527 -0.46012658 0.8192965 -0.3203064 0.7068778 -0.48808104 -0.64042336 -0.5980048 0.6758183 -0.43241805 0.3817204 -0.60529774 0.7954349 0.515999 1.0448519 -0.57644504 -0.7998966 0.4365968 -0.602568 -0.63077164 0.5883133 0.62569976 -0.30445012 -0.69754374 0.787964 0.47818944 -0.68238187 -0.4725673 -0.4951682 0.6165755 -0.41221517 -0.7899115 0.58996284 0.6890543 -0.32079244 -0.5656836 -0.48265114 -0.42626956 0.59377885 0.44245607 0.674129 -0.44998705 -0.50338304 0.6521138 0.6310461 -0.7523221; 0.48453066 0.42834747 0.737808 -0.6995326 -0.65470105 0.5641814 -0.7186385 0.6610849 -0.5679052 -0.606271 0.5590272 -0.40212452 0.46647805 0.6967276 0.6545406 -0.603067 -0.536653 0.714389 0.578849 0.51502407 0.9053242 0.6654367 -0.47712058 -0.73120004 0.075975195 -0.8249946 0.46819445 -0.5545006 -0.6642573 0.42027718 -0.28149882 -0.4364838 0.57268614 0.5302224 -0.73913604 -0.7020902 -0.35643375 -0.6349363 -0.68247914 -0.1424131 0.7246309 0.5527075 -0.45425275 0.74473995 -0.7582362 -0.11392626 0.4410645 -0.5983011 -0.4789328 -0.7490214 -0.53967524 -0.48106423 0.83375263 0.8457554 0.47495794 -0.6559901 0.010693761 -0.5694792 0.7326064 -0.56041133 0.71140337 0.7017417 -0.0044725467 -0.7236038 -0.023618136 0.38613126 -0.32427248 -0.35536572 -0.37633035 -0.37808222 0.7419138 0.42422304 0.40920168 -0.48729536 -0.514452 0.3698714 0.4299359 -0.83164805 -0.8143488 0.36998472; 0.57060266 0.7176993 -0.69994766 0.59338325 0.69411755 -0.5856974 -0.8049261 0.44511816 0.6198292 0.70894474 -0.4041475 -0.64965177 0.414722 -0.44159073 -0.6820993 -0.381442 0.4968295 -0.7578958 0.7360915 -0.49564117 -0.43621588 0.54243594 0.65172535 -0.7079791 0.49395284 0.59655774 0.54729015 -0.8213516 0.44735175 0.4584214 -0.48709735 -0.5651527 0.68149656 -0.643291 0.6199135 -0.6982265 0.62225777 -0.5951695 0.7825879 -0.75341856 -0.5463273 -0.57886845 0.5762477 -0.61934364 0.5584007 -0.6619046 0.7713038 0.524176 -0.7781785 -0.5729149 -0.7287515 0.67029345 -0.6196668 -0.5378856 0.53507465 0.5040981 -0.65418977 -0.40570843 0.46083388 0.7616376 -0.7664315 -0.5125128 -0.60826415 0.6817154 -0.5293728 -0.6871883 0.7438924 0.7178727 -0.6210035 -0.647473 -0.7172356 -0.74533856 0.7094522 0.63820297 0.75799644 -0.5038768 -0.708752 0.6816119 0.5729585 -0.69047606; 0.49949154 0.56172144 -0.50138384 0.77357334 0.4935823 -0.62211484 0.8763702 0.42503703 0.35273603 0.69795215 -0.57081413 -0.59498775 0.5820852 -0.705114 -0.38730448 0.4477829 0.6005003 -0.58146113 0.6562283 -0.8417493 -0.7698046 0.44332072 0.29516092 0.261971 0.45316967 0.8520236 0.42427868 -0.7674942 0.6195881 0.85842454 -0.8259128 0.243356 0.54218423 -0.83498025 0.62475085 0.22171558 0.6705549 0.13458222 0.83422977 -0.63638043 -0.6578201 -0.5423171 0.34120432 -0.54906934 0.56551015 -0.89541847 -0.92036146 0.7308805 0.40315503 -0.5984176 0.6086903 0.6613113 -0.83306175 -0.70816654 0.45166647 0.41325164 -0.7856451 -0.5652451 -0.79905534 0.48891297 -0.54079247 -0.87728417 -0.65791816 0.52352685 -0.22683333 -0.61843747 0.33378378 0.6617241 -0.8716288 -0.6277597 -0.64485174 -0.57143515 0.7102768 0.8259097 0.66432524 -0.88099676 -0.5219566 0.706119 0.6698619 -0.43420395; -0.647225 -0.6546222 0.45899838 -0.70443285 -0.7347124 0.7738094 0.8432064 -0.8040021 -0.67653817 -0.67738885 0.6874288 0.48691905 -0.6070119 0.6983798 0.5963185 0.56860596 -0.6013491 0.69962585 -0.5282074 0.61501867 0.28349224 -0.60448205 -0.51367795 0.63899434 -0.7496154 -0.8091007 -0.4507906 0.60455954 -0.5894151 -0.7555401 0.57763284 0.45306537 -0.7150363 0.4885167 -0.7175545 0.41753516 -0.75958383 0.43343225 -0.5144783 0.6271073 0.610622 0.44847852 -0.60513425 0.6793644 -0.52060103 0.5018703 -0.70054513 -0.5658558 0.47432157 0.65104634 0.7044092 -0.5257119 0.7481291 0.70062375 -0.8004277 -0.6632684 0.32900026 0.46671286 -0.82528776 -0.43602157 0.3702223 0.5652088 0.5695391 -0.84409034 0.3770817 0.6714308 -0.45761874 -0.6997882 0.5704443 0.79945534 0.63314337 0.6178204 -0.47157785 0.7807118 -0.59567535 0.39388353 0.8142897 -0.6491888 -0.3864936 0.8111413; -0.4598906 -0.6341269 0.70030016 -0.47324216 -0.5762474 0.44886795 0.20324458 -0.50586677 -0.6927276 -0.6250119 0.7076243 0.6359496 -0.7200306 0.604478 0.78284246 0.46755537 -0.6693464 0.5584807 -0.6514859 0.7223743 0.6505993 -0.7505584 -0.6008421 0.6414803 -0.6547714 -0.5163974 -0.47992346 0.66640776 -0.46975985 -0.57861274 0.5660194 0.5901908 -0.7423716 0.70180166 -0.75145113 0.4549112 -0.7748667 0.675779 -0.60497624 0.4204692 0.45410794 0.5074157 -0.696355 0.42482615 -0.79018074 0.49598268 -0.6345548 -0.8161526 0.45111108 0.5168592 0.57494295 -0.76556766 0.68616766 0.5786713 -0.49635723 -0.6074651 0.5314779 0.5735105 -0.71604085 -0.70868725 0.5324215 0.70589435 0.6720162 -0.7298593 0.59197813 0.4948562 -0.59079826 -0.58834565 0.69638443 0.6669331 0.5253093 0.5469091 -0.48569477 -0.7354726 -0.41652685 0.7935706 0.43532968 -0.6077922 -0.7670297 0.66497517; 0.6678511 0.4770898 -0.50812507 0.4617539 0.6058104 -0.6150042 -0.4718173 0.73431 0.36039457 0.718277 -0.6790938 -0.7240308 0.41885024 -0.6386318 -0.7150369 -0.65742457 0.80786604 -0.57857305 0.47875684 -0.69808936 -0.58124626 0.536955 0.70252824 -0.4962322 0.6084036 0.71980643 0.7179635 -0.4896413 0.7220635 0.60253894 -0.4653253 -0.49805453 0.70685416 -0.6403717 0.60025376 -0.5554808 0.48176897 -0.4943467 0.6955696 -0.5636342 -0.5920132 -0.4988754 0.39913172 -0.54434144 0.59709066 -0.5209335 0.42279628 0.82729834 -0.5126687 -0.5984578 -0.44577333 0.7589182 -0.54197717 -0.81616974 0.4580172 0.52285504 -0.59812933 -0.40350693 0.7435836 0.57051885 -0.5810215 -0.8091958 -0.68809104 0.4168723 -0.61786234 -0.5768029 0.7349241 0.40152922 -0.563167 -0.7240045 -0.47821096 -0.56563526 0.5756279 0.8613174 0.63967353 -0.84255457 -0.47121903 0.6200834 0.8166073 -0.71997935; 0.5345879 0.7000338 -0.7245442 0.53702736 0.4831735 -0.4839272 0.6961218 0.6636424 0.6257728 0.42924967 -0.71170986 -0.6749976 0.40774482 -0.71628493 -0.5703459 0.32205254 0.75380385 -0.56298137 0.5758289 -0.5087041 -0.45558617 0.6933969 0.6483019 -0.13752401 0.70420796 0.6714673 0.71367323 -0.839659 0.6792832 0.4953628 -0.67459506 -0.9461765 0.67201734 -0.59563076 0.6169371 -0.72208315 0.45073333 -0.6919946 0.5543506 -0.5703606 -0.49185452 -0.6646332 0.7444688 -0.47563782 0.6837722 -0.8290913 0.62874496 0.6512275 0.029114109 -0.5478497 -0.2056927 0.6807715 -0.6645758 -0.51361126 0.5880299 0.5395352 -0.7167637 -0.7014543 0.38281608 0.8051706 -0.504172 -0.6503848 -0.4470974 0.5935619 -0.42939135 -0.6606353 0.5729221 0.6823059 -0.7124986 -0.7408094 -0.5693795 -0.69101715 0.62500155 0.5307194 0.6800185 -0.6993853 -0.6784083 0.6225363 0.5765532 -0.6404077; 0.5692614 0.7304097 -0.4571553 0.6308211 0.5480558 -0.5889523 -0.55239224 0.4843653 0.4371375 0.5616033 -0.62335306 -0.4574996 0.52666754 -0.7601381 -0.61660045 -0.7968783 0.46286947 -0.52662665 0.746028 -0.5449526 -0.76470864 0.7786726 0.59218526 -0.662857 0.597623 0.76517063 0.6869156 -0.5104293 0.53429544 0.46837318 -0.745979 -0.6851201 0.45428035 -0.46681476 0.7451405 -0.48700976 0.6792489 -0.695316 0.49395815 -0.40948063 -0.52325153 -0.6674665 0.56455064 -0.5413808 0.76135135 -0.42812216 0.6600149 0.44046795 -0.5019492 -0.6654703 -0.61416405 0.6263462 -0.6410098 -0.43201038 0.7049235 0.577488 -0.5753567 -0.5540606 0.7686534 0.474647 -0.69229406 -0.42992246 -0.5792896 0.39052457 -0.62876487 -0.73835206 0.66312474 0.47588664 -0.7885468 -0.76519185 -0.4863976 -0.44369385 0.68632025 0.4787453 0.7667649 -0.5374372 -0.68788403 0.68631923 0.5983305 -0.47384462; 0.6516377 0.539224 -0.6008042 0.6309223 0.64590335 -0.66851807 -0.8896641 0.49916264 0.6695176 0.6984722 -0.38841444 -0.55920374 0.6373687 -0.6932866 -0.4307824 -0.3124081 0.67329824 -0.78207433 0.75730807 -0.6488122 -0.66776514 0.6335892 0.7058241 -0.7759141 0.55949044 0.6582508 0.66525584 -0.63364404 0.60785747 0.62469476 -0.5823225 -0.70658 0.47935736 -0.7748788 0.60606205 -0.773842 0.54835343 -0.49406245 0.51119924 -0.44232297 -0.49080765 -0.44228265 0.73555136 -0.62131107 0.56306434 -0.6350579 0.74200803 0.49857718 -0.41016516 -0.7044394 -0.4062185 0.63796896 -0.52300614 -0.7496344 0.6028626 0.68819034 -0.6529806 -0.4654534 0.7549601 0.56343865 -0.76861495 -0.58449966 -0.59359205 0.56504023 -0.28224868 -0.7700049 0.46878293 0.44278902 -0.5753081 -0.75967485 -0.45479876 -0.6428229 0.4602817 0.6898457 0.68815136 -0.5857992 -0.51062685 0.73072165 0.5320299 -0.75146073; -0.62060297 -0.55263644 0.698015 -0.4793442 -0.6415646 0.40559226 0.58216405 -0.40445355 -0.3755788 -0.5313756 0.62222177 0.5453518 -0.6020272 0.53628576 0.45428655 0.48806664 -0.57581776 0.4183632 -0.43528548 0.735252 0.5877097 -0.5860946 -0.71136284 0.51951474 -0.7408845 -0.7513116 -0.54507214 0.56011474 -0.5252322 -0.75276625 0.7307821 0.6450318 -0.43061128 0.757803 -0.5423339 0.7214321 -0.72719234 0.7560937 -0.5420154 0.48034254 0.41139102 0.50451547 -0.62118596 0.65055907 -0.43567747 0.6815154 -0.4734455 -0.4338187 0.69290257 0.69782215 0.49204654 -0.42819786 0.4041776 0.5254171 -0.6584771 -0.57769793 0.4931713 0.51931584 -0.61722624 -0.50289583 0.4853386 0.4230517 0.48638427 -0.52587533 0.797826 0.22375359 -0.510465 -0.42978725 0.454171 0.6351077 0.5467436 0.4026516 -0.60774565 0.5596501 -0.27544773 0.5311648 0.6543231 -0.70152444 -0.7193279 0.68395364; 0.53832984 0.5651947 -0.40484527 0.6911873 0.48387867 -0.7114959 -0.37909296 0.42562985 0.7540656 0.43653047 -0.7515136 -0.7131207 0.4790264 -0.5465739 -0.671148 -0.7131588 0.754644 -0.5534639 0.67695814 -0.4595282 -0.5072386 0.70082635 0.6528742 -0.7739104 0.4361298 0.56547695 0.7605496 -0.7744045 0.62991315 0.6544029 -0.46553922 -0.6138525 0.4245465 -0.5009826 0.71423244 -0.753915 0.7578102 -0.5495706 0.40341008 -0.46697244 -0.77122384 -0.5536904 0.48331818 -0.46357825 0.41000354 -0.467669 0.64203256 0.45414588 -0.5552934 -0.44223544 -0.7065273 0.47774982 -0.41146737 -0.45028436 0.4547479 0.53876084 -0.6812385 -0.79060507 0.49420163 0.4582821 -0.47060114 -0.756638 -0.6561157 0.45525944 -0.55286956 -0.73792857 0.64772254 0.5745922 -0.5402572 -0.78413314 -0.40816355 -0.63769114 0.6203711 0.72852975 0.7097858 -0.44069684 -0.7935208 0.42584026 0.5990746 -0.7866587; -0.4583934 -0.67672545 0.7965037 -0.66747695 -0.6802312 0.70894974 0.8406911 -0.7622663 -0.9770816 -0.70142907 0.5769551 0.618773 -0.46429235 0.5384925 0.6677434 0.3756659 -0.68020755 0.553973 -0.44723383 0.38720858 0.6169596 -0.8225392 -0.6348862 0.41016796 -0.46788868 -0.5552633 -0.5527563 0.49006656 -0.5071302 -0.6773529 0.66962147 0.36988512 -0.51799285 0.53339565 -0.48846862 0.6054092 -0.80390674 0.4572823 -0.5034225 0.5707822 0.57984495 0.65797395 -0.7867344 0.41727066 -0.76440567 0.6800209 -0.69438505 -0.34003776 0.5691956 0.64324737 0.8311628 -0.3939693 0.50368804 0.48050714 -0.67750585 -0.83421516 0.5799126 0.72266525 -0.6488696 -0.74830955 0.57084006 0.57371366 0.62626696 -0.8526508 0.32967117 0.9150749 -0.49118105 -0.61657655 0.51563436 0.52971756 0.51790065 0.7206795 -0.7205679 1.0195442 -0.6708519 0.56118995 0.57180965 -0.56296927 -0.64010507 0.5709438; 0.81776285 0.59011286 -0.4043668 0.54781866 0.75155556 -0.67979157 -0.7210113 0.56835127 1.1270968 0.5224012 -0.79118675 -0.42583635 0.62845814 -0.5445534 -0.6508665 -0.56953716 0.37175673 -0.44923666 0.6739148 -0.67157656 -0.67422646 0.5416562 0.69853723 -0.36575353 0.50988716 0.3703307 0.4809554 -0.4261198 0.6371862 0.5549134 -0.5289853 -0.7699877 0.52102727 -0.601154 0.69319475 -0.4350058 0.76398957 -0.56421036 0.5450566 -0.61782765 -0.51029056 -0.49975285 0.7107429 -0.48174495 0.47538304 -0.42410305 0.64686525 0.22160305 -0.52242833 -0.47123298 -0.5752993 0.54534537 -0.59492284 -0.51558024 0.7900153 0.5357832 -0.46919596 -0.5273621 0.7002913 0.68113923 -0.59304065 -0.49666116 -0.75335133 0.87238866 -0.70373154 -0.25610983 0.6261517 0.83046883 -0.5053519 -0.8070039 -0.4138931 -0.79196113 0.6043999 -0.5649351 0.7329235 -0.39562657 -0.6635965 0.4881566 0.44722572 -0.5384898; -0.8143877 -0.5388987 0.5622834 -0.48663962 -0.59331876 0.5631666 -0.5367703 -0.63822246 -0.8525555 -0.6884294 0.52100414 0.72006613 -0.6528122 0.56273484 0.7827213 -0.7554238 -0.34672597 0.714805 -0.5391306 0.589576 1.2298608 -0.64438653 -0.49628773 -0.36484268 -0.7017778 -0.37845114 -0.7250952 0.6392222 -0.49846503 -0.5797044 0.44173178 0.5354788 -0.5086914 0.5925052 -0.3769601 0.5383718 -0.82004446 0.5344789 -0.43316656 0.60296726 0.39093432 0.34943 -0.6105097 0.6078849 -0.49614674 0.46874318 -0.56614363 -0.6479248 -0.6470893 0.71601546 0.47172388 -0.7166457 0.477152 0.4138147 -0.59190226 -0.6734313 0.54234934 0.49076757 -0.43145686 -0.3519652 0.4535293 0.61498785 0.41076267 -0.7325534 0.4373698 0.81636506 -0.7219137 -0.6716097 0.5795255 0.6928355 0.33462095 0.8649462 -0.3989296 -0.6920157 -0.8351687 0.75994146 0.66088206 -0.42561466 -0.42538548 0.61989874; -0.34560692 -0.6080449 0.6581993 -0.6676557 -0.58602726 0.7104754 0.8079094 -0.79076433 -0.5811597 -0.56700706 0.42258734 0.7410489 -0.39466396 0.45036635 0.69105655 0.32041755 -0.82518446 0.47881576 -0.7539561 0.59346855 0.5316065 -0.648807 -0.5001008 0.727232 -0.6866964 -0.5103273 -0.59563154 0.5426506 -0.6713982 -0.594624 0.4780682 0.7507751 -0.68638635 0.75284606 -0.5331657 0.46949 -0.7040477 0.76305664 -0.68576854 0.51509106 0.80598545 0.78739136 -0.7144851 0.7121072 -0.4913868 0.7149437 -0.5381577 -0.4451061 0.43113932 0.6816542 0.5923237 -0.4682287 0.8062472 0.552164 -0.7170948 -0.41127038 0.68575436 0.49880326 -0.6212397 -0.69206816 0.7984349 0.61806875 0.77134544 -0.3816164 0.6459509 0.5283604 -0.7091562 -0.7396544 0.54603076 0.54247767 0.58681256 0.5458619 -0.6645362 -0.51672727 -0.5287939 0.70224875 0.5509831 -0.57214135 -0.61605215 0.46250635; -0.61313456 -0.39904156 0.50781864 -0.7338908 -0.70388347 0.76397663 -0.4722954 -0.55971515 -0.47612908 -0.7321419 0.6205881 0.6371874 -0.7143448 0.54645205 0.7080509 -0.34204668 -0.5464163 0.69854015 -0.77880776 0.71288085 0.47841847 -0.62825966 -0.74203724 0.56160223 -0.5886748 -0.46749616 -0.5941465 0.5629584 -0.40551364 -0.45828548 0.7115675 0.5311641 -0.5044927 0.48947203 -0.5661513 0.5858769 -0.6859715 0.64878464 -0.5898578 0.514994 0.709164 0.6161517 -0.6168769 0.59311426 -0.7401917 0.8024996 -0.4761477 -0.6016425 0.63879544 0.38169125 0.6572181 -0.6961573 0.47032169 0.4669378 -0.5295524 -0.6808322 0.77276266 0.72802496 -0.6738825 -0.4931109 0.58952886 0.59955645 0.50484127 -0.6180407 0.599072 0.6192596 -0.5348084 -0.47363257 0.75374067 0.7346297 0.71411556 0.73527735 -0.7887357 -0.6143097 -0.7321353 0.51084185 0.7612971 -0.4869964 -0.47475168 0.77424854; -0.5050571 -0.69943523 0.49693784 -0.4785752 -0.45662147 0.40399256 0.65735006 -0.53476095 0.3957023 -0.7025972 0.71609634 0.7148353 -0.5406652 0.80119234 0.626865 0.70860624 -0.63114154 0.6794215 -0.6545873 0.5356104 0.7812936 -0.4545135 -0.6259596 0.5074256 -0.5825523 -0.61137426 -0.4673174 0.5338991 -0.6340026 -0.6375981 0.71399117 0.64971125 -0.44975647 0.57878405 -0.6922524 0.5502553 -0.4520423 0.7421752 -0.4891123 0.66207737 0.49396554 0.77600104 -0.567734 0.5834411 -0.79797465 0.67471904 -0.7496933 -0.36937416 0.81078285 0.67869496 0.49053738 -0.7102923 0.48147354 0.5094612 -0.69915783 -0.7062988 0.47281155 0.54276973 -0.47804216 -0.44842127 0.46464422 0.56965667 0.4834419 -0.6980087 0.59928334 -0.718939 -0.5096134 -0.45698795 0.6877343 0.58693796 0.553327 0.59267825 -0.5461445 0.48242095 0.6102762 0.5358631 0.7053317 -0.5203708 -0.37505296 0.7125253; -0.29906425 -0.6026751 0.6918526 -0.70870656 -0.6903183 0.54667103 -0.7918507 -0.5859286 -0.42111176 -0.4560768 0.49283403 0.6986537 -0.49244022 0.7002288 0.43867356 -0.40297577 -0.8295366 0.69102883 -0.5872638 0.79356486 0.667844 -0.31061047 -0.58273506 -0.27362183 -0.53678775 -0.8242279 -0.7104472 0.85120887 -0.6005164 -0.7657788 0.8535119 -0.40822902 -0.66062564 0.57908773 -0.6838193 -0.24905302 -0.33440208 -0.026904307 -0.6618595 0.884149 0.56234586 0.74788386 -0.34610754 0.7801059 -0.8243648 0.5897485 0.18742034 -0.6699728 -0.43777686 0.5233283 -0.9362473 -0.88705575 0.5775784 0.66595906 -0.56999415 -0.444607 0.66430455 0.5226047 0.43194696 -0.76510596 0.568613 0.76523143 0.81781447 -0.48587295 0.5238945 0.39627135 -0.66518 -0.40325728 0.8306141 0.39991093 0.6889143 0.44628084 -0.8054792 -0.7139685 -0.40232712 0.84883595 0.64360046 -0.73586863 -0.56543815 0.49407512; 0.46665105 0.38969412 0.80849266 -0.8068379 -0.7158155 0.61607665 -0.5435307 0.5169046 -0.68558234 -0.84086674 0.53765833 -0.6401472 0.6190705 0.6875482 0.41041896 -0.44722572 -0.39760214 0.5863697 0.4268506 0.7749094 0.7867522 0.55250716 -0.7344884 -0.74270624 0.46449524 -0.6927962 0.47228923 -0.6920799 -0.68161297 0.48829898 -0.6705132 -0.55363166 0.60868233 0.66652364 -0.6612312 -0.5749568 -0.4221564 -0.45359653 -0.5495402 -0.6076216 0.68953377 0.8651579 0.35934085 0.64069223 -0.7992981 -0.2904761 0.72624815 -0.68063295 -0.5432729 -0.44960046 -0.6927173 -0.67588013 0.36366773 0.84880584 0.4172692 -0.46121296 -0.48272592 -0.6998583 0.6534302 -0.6931555 0.47143668 0.53666395 -0.5815005 -0.51040554 -0.16717157 0.72223794 -0.5254851 -0.4507502 -0.52937484 -0.42274618 0.5980053 -0.45501232 0.5524582 -0.5463172 -0.4908598 0.18769394 0.7309498 -0.8054575 -0.74761426 -0.29768094; -0.6127519 -0.7897234 0.42755452 -0.7025863 -0.727623 0.62837166 -0.10820207 -0.84825015 -0.7282995 -0.8819426 0.59631294 0.7317749 -0.8405239 0.5855841 0.70254856 0.26499403 -0.593932 0.3828341 -0.7346612 0.6161663 0.7335692 -0.74724627 -0.7309292 0.2438446 -0.9007341 -0.458405 -0.6883608 0.3992729 -0.85941 -0.4698977 0.6762296 0.48866007 -0.60858864 0.42324445 -0.41925693 0.27047852 -0.5163495 0.40446147 -0.6247474 0.6492819 0.57789844 0.43891606 -0.69715136 0.36749378 -0.45851403 0.55787444 -0.7482548 -0.4067355 0.606831 0.6382958 0.5184106 -0.67630893 0.4043663 0.5429243 -0.76608056 -0.82976234 0.18209372 0.8656683 -0.3970933 -0.62806606 0.4362761 0.54469085 0.50256884 -0.7282202 0.15629134 0.77414435 -0.76916784 -0.62933046 0.5813968 0.57826203 0.42268944 0.6834607 -0.4728204 -0.44716752 -0.5425279 0.5981676 0.7167171 -0.43631515 -0.6027464 0.6274444; -0.75595915 -0.594664 0.4419684 -0.66219866 -0.5923647 0.5756262 0.7344823 -0.5896353 -0.62141806 -0.4805518 0.41453263 0.44078496 -0.6215154 0.6036975 0.44889402 0.7028329 -0.71005595 0.64785826 -0.6007164 0.53525734 0.70464027 -0.4280571 -0.41178015 0.5308606 -0.65893483 -0.5615179 -0.74456936 0.6492469 -0.5827944 -0.75149035 0.5502461 0.73489994 -0.7357721 0.50974345 -0.47950104 0.46266496 -0.40536836 0.4936884 -0.5695652 0.52450156 0.48320088 0.73510855 -0.68940896 0.5469481 -0.41088402 0.45640722 -0.64540386 -0.47288778 0.7081609 0.5078539 0.6972777 -0.6084809 0.5757455 0.7209497 -0.56587774 -0.7806526 0.4462064 0.6678558 -0.6290313 -0.5317173 0.56552976 0.7330277 0.6305286 -0.66700786 0.65826064 0.44812623 -0.73249745 -0.6279158 0.7920457 0.63970906 0.41440466 0.7193952 -0.51449656 -0.8261638 -0.72743064 0.42443755 0.6294433 -0.4087143 -0.5009509 0.6218808; 0.6387771 0.6371576 -0.69393516 0.546034 0.63025576 -0.8789776 -0.671862 0.5484431 0.6404824 0.7781044 -0.6201823 -0.82806295 0.8511346 -0.5106093 -0.8081617 -0.4919564 0.4811388 -0.740494 0.7588527 -0.66575295 -0.5596076 0.5113973 0.549537 -0.48653454 0.60087097 0.46179926 0.63406074 -0.37873748 0.56455535 0.54672277 -0.3541641 -0.43196532 0.5639576 -0.6464712 0.5031362 -0.6652226 0.830604 -0.4183454 0.72698325 -0.4117347 -0.69828373 -0.783787 0.48252198 -0.4080174 0.6598216 -0.48560742 0.8560065 0.46136945 -0.49013567 -0.5765095 -0.81788135 0.35554293 -0.4187475 -0.7153291 0.8172138 0.7983016 -0.24997991 -0.7029839 0.5185781 0.7213711 -0.4157557 -0.7608783 -0.577318 0.57624197 -0.2595923 0.07080421 0.83775496 0.6202469 -0.36519155 -0.5712256 -0.5293601 -0.4905436 0.5243545 -0.81154823 0.77890104 -0.39704806 -0.84728134 0.6611374 0.24888778 -0.62835497; -0.5476642 -0.6718965 0.49460068 -0.69903547 -0.5916054 0.4411067 -0.7011732 -0.44547483 -0.5574753 -0.4721002 0.6778876 0.44769472 -0.6694683 0.45509136 0.7483606 -0.5071351 -0.6204986 0.5363274 -0.4455256 0.46592423 0.636559 -0.6094425 -0.4507876 0.78214484 -0.5470007 -0.51898605 -0.5997515 0.77015436 -0.65204895 -0.6647835 0.6741864 0.75316954 -0.6722167 0.4196016 -0.6268611 0.5610859 -0.46230185 0.5614477 -0.68265694 0.44358647 0.7690558 0.4864964 -0.4593457 0.4260133 -0.7036911 0.48098055 -0.5223405 -0.7541645 0.6734015 0.44156128 0.6724313 -0.71089 0.4164007 0.58177793 -0.49716213 -0.5648564 0.7661503 0.7751865 -0.6287634 -0.52994514 0.6046695 0.6507389 0.5947433 -0.48145762 0.7091179 0.6780848 -0.72915685 -0.74713933 0.69679743 0.5087339 0.702943 0.75355154 -0.5319134 -0.62916297 -0.7284612 0.6164651 0.6490074 -0.5361723 -0.49701813 0.6414342; 0.42152387 0.6226024 -0.5490915 0.5950689 0.48439038 -0.4888427 0.6177657 0.7133461 0.41315538 0.5943653 -0.40347052 -0.42108974 0.6759178 -0.62589943 -0.606116 0.42744 0.68176645 -0.7149724 0.4039765 -0.59109014 -0.40677193 0.68429637 0.7289715 -0.94029546 0.55780107 0.58613443 0.57595676 -0.83609515 0.5659482 0.6770302 -0.80683804 -0.76380044 0.7537821 -0.471903 0.61835927 -0.511491 0.72899944 -0.6341839 0.73078686 -0.4290447 -0.5127789 -0.60535944 0.7706206 -0.4958763 0.6152139 -0.7885552 0.48398903 0.601404 -0.66346747 -0.44403446 -0.5060254 0.4951007 -0.42449662 -0.5459631 0.577544 0.396965 -0.88573873 -0.73347205 0.8254257 0.71011204 -0.6678458 -0.639335 -0.62164736 0.38093117 -0.692741 -0.44838026 0.42182907 0.6758161 -0.5286312 -0.5988335 -0.42219916 -0.5344236 0.53352845 0.754878 0.5196139 -0.577684 -0.4806967 0.51338005 0.5228737 -0.41130206; 0.77049285 0.5258762 -0.78072 0.60410935 0.7628258 -0.41337067 -0.7889723 0.70113593 0.65610147 0.7629184 -0.46823174 -0.59791183 0.7490983 -0.44208604 -0.64069605 -0.52624565 0.57621557 -0.54358625 0.7170993 -0.55951995 -0.67458105 0.6639799 0.52416974 -0.43703118 0.76551193 0.7862601 0.45264325 -0.47537065 0.64237106 0.5788352 -0.59733886 -0.7539731 0.7290927 -0.69252187 0.4329777 -0.53721607 0.7462543 -0.5220364 0.6078356 -0.6898347 -0.6223551 -0.6026936 0.6068065 -0.5936956 0.53958553 -0.4565934 0.6569296 0.67420435 -0.54783285 -0.63056785 -0.55211955 0.5456263 -0.79256535 -0.7256031 0.6313464 0.5026736 -0.40453258 -0.75649554 0.4404553 0.6527122 -0.44336987 -0.6932745 -0.73746717 0.4745765 -0.5393345 -0.7249673 0.6644542 0.7300395 -0.43531042 -0.4519844 -0.61304975 -0.53865063 0.47081283 0.560561 0.51382273 -0.5218821 -0.43319383 0.539053 0.6561517 -0.6147021; -0.75754917 -0.4376816 0.5558849 -0.7079664 -0.72527826 -0.7478429 0.58015496 -0.68482107 0.771549 -0.40785688 0.40280232 0.4953298 -0.54815066 -0.867661 0.17931907 0.45577392 -0.682036 0.3905842 -0.42848757 -0.57466155 0.647153 -0.4499851 0.49650094 0.7700811 -0.67372733 -0.58793044 -0.54968566 0.783568 -0.39303997 -0.56855804 0.7411747 0.6230472 -0.6306065 0.4400098 -0.62629044 0.41390523 0.33407533 0.7562734 0.8871244 0.48398745 0.50003695 0.49284473 -0.46687794 0.36521986 -0.5187344 0.6047532 -0.5337434 0.6093278 0.7308699 0.5195639 0.56704015 0.7114109 0.7400147 -0.49646458 -0.55694425 -0.49157146 0.6532415 0.6028483 -0.62681794 -0.18161023 -0.629528 0.7534123 0.49386835 0.47383487 -0.07503049 -0.61298835 -0.46166423 -0.3060985 0.6160743 0.6493677 0.6614936 0.64179623 -0.67637706 0.7027141 0.44726092 0.5380469 0.68858993 -0.37928402 0.584743 0.44266215; -0.7447432 -0.7203492 0.7047483 -0.7318692 -0.57744884 0.64322567 0.58044654 -0.69194436 0.028003404 -0.59246546 0.64358234 0.78740746 -0.6022731 0.5661569 0.58258283 0.5148686 -0.47533846 0.57524323 -0.61924314 0.46153864 0.58867794 -0.57819784 -0.56853825 0.54819554 -0.5372691 -0.48525763 -0.6308935 0.5876487 -0.7107062 -0.63629246 0.60317755 0.6016365 -0.46472955 0.6846745 -0.76289606 0.50369996 -0.7195571 0.43154445 -0.42836583 0.7755051 0.5902027 0.50319993 -0.47113025 0.73465073 -0.47088882 0.5141756 -0.49013984 -0.15855904 0.62038976 0.49597976 0.57192296 -0.5339556 0.6170308 0.5543698 -0.5832519 -0.47846448 0.45745248 0.7027594 -0.44526756 -0.5665158 0.5789255 0.7270715 0.5568856 -0.53150254 0.35044986 -0.42726246 -0.50604457 -0.7986301 0.6600166 0.58981764 0.7615775 0.56506 -0.67934304 0.8524071 -0.08668141 0.471311 0.5524299 -0.6955303 -0.4609339 0.6760539; 0.51159674 0.5372108 -0.59609073 0.5131836 0.5887139 -0.4131295 0.79965794 0.7504149 0.38985407 0.6716658 -0.39254153 -0.5133184 0.39025083 -0.53687 -0.4427081 0.48777363 0.6386426 -0.5749398 0.45107096 -0.52514905 -0.44247884 0.6946687 0.75256103 -1.020414 0.47470596 0.4393298 0.5931582 -0.73032975 0.52346075 0.6916331 -0.7784707 -0.5365013 0.7426783 -0.77870166 0.5288837 -0.7657967 0.5961319 -0.7359674 0.49786997 -0.6446817 -0.60543275 -0.74408674 0.47390518 -0.65883523 0.68752193 -0.78815186 0.5338453 0.60560626 -0.70232636 -0.58654934 -0.3666893 0.41650686 -0.51763976 -0.74668133 0.7435176 0.6726043 -0.7464552 -0.7199824 0.8058171 0.4681344 -0.5046763 -0.48636842 -0.769215 0.56009126 -0.67677623 -0.7429624 0.40950614 0.5749519 -0.73121506 -0.66831994 -0.71389323 -0.49468467 0.47375757 0.60833853 0.7197339 -0.48869985 -0.6580996 0.4281358 0.66257375 -0.44221088; 0.4886417 0.663129 -0.6137113 0.7172794 0.54455864 -0.5753674 -0.7470428 0.36369663 0.45552003 0.6055759 -0.7385137 -0.6033203 0.62502044 -0.43329927 -0.46355456 -0.4173644 0.38442135 -0.481911 0.72680795 -0.6608814 -0.6618418 0.52598804 0.5103756 -0.39378402 0.6248374 0.44809377 0.42220116 -0.5905372 0.72259 0.42969534 -0.521738 -0.7230644 0.7729947 -0.54800475 0.4944953 -0.62607527 0.77740914 -0.54592645 0.48554182 -0.73716086 -0.54130507 -0.64421237 0.7480223 -0.43070745 0.46985036 -0.51150405 0.64024544 0.53020114 -0.6868734 -0.72923976 -0.5027815 0.4412877 -0.72689533 -0.5775606 0.44537172 0.65463156 -0.54979676 -0.5522508 0.5688013 0.40626818 -0.5008265 -0.51672125 -0.58783257 0.7527343 -0.611411 -0.5509025 0.55686426 0.76677036 -0.6842918 -0.4275884 -0.51674825 -0.6340506 0.48883164 0.49477115 0.76645356 -0.57782894 -0.59862566 0.5095041 0.63175 -0.6495483; 0.4916806 0.66751397 -0.74954575 0.5601284 0.62475973 -0.6951622 0.69088507 0.6602517 0.47829655 0.65234756 -0.7526971 -0.59162706 0.6193215 -0.7251101 -0.72943157 0.5757666 0.5633241 -0.54574305 0.477736 -0.7656442 -0.63566303 0.62551534 0.64852047 -0.5550583 0.41931835 0.47289252 0.6769652 -0.48047498 0.68483484 0.52235067 -0.7613736 -0.5861101 0.44372943 -0.4326988 0.49300045 -0.53933513 0.7027391 -0.44425294 0.5331082 -0.5666726 -0.64316064 -0.4239321 0.53230083 -0.74292004 0.63220435 -0.7134532 0.47830436 0.446497 -0.72054625 -0.7075487 -0.40608427 0.43450898 -0.5471602 -0.46066532 0.6930468 0.6274419 -0.5782891 -0.6042913 0.52341616 0.5802491 -0.6705543 -0.5909606 -0.67823994 0.76842976 -0.65803456 -0.43645737 0.6845035 0.6808328 -0.77988076 -0.44352177 -0.45280445 -0.68075496 0.48740602 0.5254221 0.6157094 -0.5350816 -0.4082149 0.43136737 0.5462216 -0.74202526; 0.6038308 0.54974484 -0.5812858 0.6014858 0.37873155 -0.90969366 -0.7338479 0.77608424 0.9949548 0.58502465 -0.8864965 -0.7528902 0.57548624 -0.7003966 -0.56819123 -0.61943746 0.46914586 -0.55267584 0.67174196 -0.60355127 -0.3374376 0.55878425 0.5830177 -0.63578033 0.789682 0.5545334 0.70726573 -0.6056816 0.720203 0.6484663 -0.38898563 -0.5985198 0.6275905 -0.6216699 0.58471406 -0.44493753 0.6781862 -0.5839938 0.312493 -0.48788133 -0.43138877 -0.6568386 0.58142066 -0.58416647 0.3704562 -0.39014855 0.5890249 0.53968924 -0.51560545 -0.70045805 -0.5511054 0.3370777 -0.70403945 -0.6492574 0.68485713 0.854144 -0.24731185 -0.65093166 0.6167341 0.5446819 -0.37551147 -0.5618145 -0.3842503 0.8729769 -0.54550236 -0.76487035 0.66742176 0.86524826 -0.41856167 -0.73896086 -0.40523255 -0.45422095 0.5419758 -1.1967596 0.9503177 -0.4656733 -0.8375055 0.582759 0.5777662 -0.77904814; -0.6756805 -0.50161195 0.7764471 -0.6064978 -0.5291943 0.3609161 0.7440187 -0.50517607 0.5734324 -0.65941566 0.43693912 0.46829143 -0.58332664 0.6571085 0.61661124 0.658557 -0.6688486 0.7275786 -0.531645 1.0229458 0.69289464 -0.49776217 0.67724043 0.7136571 -0.4138719 -0.634221 -0.5906369 0.6828893 -0.39160448 -0.49007174 0.7689875 0.5814746 -0.4999317 0.630089 -0.80042326 0.53516597 -0.06840761 0.67307806 -0.77954423 0.52917117 0.5171584 0.60381615 -0.5369257 0.76189667 -0.5282256 0.5936243 -0.45247915 0.26431563 0.74624854 0.472711 0.710299 -0.7647124 0.48513812 0.8298744 -0.5500308 -0.41205212 0.7837537 0.5634672 -0.45245236 -0.5318349 -0.018574694 0.70708585 0.73923004 0.627078 -0.5247962 -0.4925442 -0.6470799 -0.24188149 0.6385963 0.643105 0.5945293 0.6250114 -0.576778 0.42763618 0.5209424 0.5691188 0.73479474 -0.67611206 0.4994221 0.66074336; -0.68436486 -0.7553965 0.6361469 -0.6175558 -0.6030789 0.7862691 0.7151502 -0.7626324 -0.6825606 -0.80416507 0.6586797 0.6666595 -0.6919416 0.43308264 0.61420256 0.6114965 -0.49192262 0.3355205 -0.6853925 0.49128857 0.5957077 -0.6738085 -0.7909487 0.5055621 -0.8504351 -0.5793267 -0.76270896 0.5660237 -0.7237019 -0.4695198 0.30204317 0.53148454 -0.6381861 0.5203495 -0.6584706 0.42823339 -0.81962377 0.4039755 -0.36742496 0.4604647 0.49687788 0.43296754 -0.7880669 0.47862923 -0.42146412 0.66054034 -0.63083667 -0.5456694 0.41509908 0.5472807 0.81771135 -0.46257383 0.57269925 0.49639255 -0.61504805 -0.7198696 0.39592102 0.457644 -0.6207133 -0.66453993 0.48197383 0.5733322 0.40928128 -0.87171906 0.61950445 0.6323505 -0.83438545 -0.8090792 0.5506281 0.6948285 0.5095852 0.63916886 -0.6403552 0.73971593 -0.893242 0.638663 0.6815576 -0.4280833 -0.5879207 0.8437456; -0.37273705 0.33197626 -0.79586065 0.8298999 0.6530267 -0.6731045 0.5617409 -0.51856 0.6461004 0.78426725 -0.3781375 -0.5863734 0.018823061 -0.7191534 -0.6233888 0.50727224 0.62830836 -0.7982342 0.44612658 -0.5911464 -0.73370683 -0.5681666 0.65211505 0.4792394 0.32032174 0.49735278 0.47251776 0.57208353 0.74838424 0.41681808 0.14899038 0.62606716 0.61465216 -0.6184528 0.79636353 0.43277597 0.53526616 0.3889616 0.8392785 -0.47804123 -0.80876404 -0.639356 0.49182758 -0.583236 0.8253439 -0.69355303 -0.57065016 0.6919762 0.6326427 0.66307247 0.7693721 0.5866766 -0.62236154 -0.6626245 -0.73170334 0.7413138 -0.2849471 0.6962302 -0.45180932 0.82436955 -0.6510369 -0.7125197 -0.68371373 0.46822578 -0.047533363 -0.48538792 0.4712512 0.35996312 0.08781689 -0.5290704 -0.85763156 -0.56544006 -0.21570317 0.5500521 0.64505494 -0.73453844 -0.6472312 0.5189621 0.7271013 -0.41111052; -0.8447003 -0.90022284 0.3058108 -0.50824517 -0.6601446 0.59725326 -0.3907906 -0.8403404 -0.8804074 -0.6966514 0.8490491 0.8213966 -0.578449 0.35864803 0.64190716 -0.84075195 -0.57594496 0.38408485 -0.8492754 0.42084354 0.56987196 -0.6257887 -0.73632306 -0.40289423 -0.8147758 -0.66377175 -0.76238275 0.32415363 -0.7985267 -0.3041185 0.40548745 0.4062813 -0.91130805 0.66940373 -0.5472192 0.46566415 -0.9341988 0.31839594 -0.49143177 0.49393037 0.6179414 0.4737422 -0.73073745 0.59316295 -0.3708839 0.50787956 -0.8712675 -0.5863126 -0.42332572 0.8261697 0.71241975 -0.4825106 0.6437626 0.5568819 -0.7540576 -0.7327534 0.46749902 0.87439585 0.1031141 -0.3616181 0.4827659 0.47431272 0.5305509 -0.9253078 0.24330582 0.7885667 -0.7464953 -0.56693745 0.5863227 0.5732733 0.6611457 0.6432743 -0.6559756 -0.46666977 -0.7677088 0.36615628 0.822539 -0.61646235 -0.28455594 0.86122066], bias = Float32[-0.6301854; -0.5675273; 0.85339755; -0.72161275; -0.89019674; 0.61417603; -0.55549186; -0.7510236; 0.59476715; 0.59357864; -0.6423533; -0.60761166; 0.58955014; 0.5919887; -0.59703624; 0.63024557; 0.62124777; 0.5851941; 0.50944096; 0.59506696; 0.5805477; 0.57747185; -0.52945006; -0.5936181; -0.5850676; 0.5256293; 0.58767635; 0.42987135; 0.7525636; 0.50725466; -0.7781706; 0.66607654; 0.53531134; -0.43843448; -0.47710547; 0.61497647; -0.52301425; 0.5715535; 0.5875541; -0.6099056; -0.6661467; 0.60613215; 0.5969688; 0.5459998; -0.6965166; 0.45137155; -0.5325554; -0.4566633; 0.6110229; 0.5859702; -0.531381; -0.5631591; -0.60237; -0.55088353; 0.63499236; -0.60066867; 0.6022974; -0.70190316; 0.57619023; 0.51296234; 0.5650644; 0.56239337; 0.4343919; 0.43660358; 0.5835632; 0.5938625; -0.6187468; 0.59222007; -0.593513; -0.5742358; 0.026098795; 0.5947856; -0.5868109; -0.6594776; -0.59644645; -0.7161568; 0.49585733; 0.60725784; -0.4662414; 0.76809067]), layer_3 = (weight = Float32[-0.5248468 0.7198479 -1.0923171 -0.5429131 0.43047315 0.5097951 -0.93780714 -0.69418883 0.39490074 0.4073934 -0.7007496 -0.43213394 0.29068705 -0.6035029 -0.33347034 0.57544947 0.5647877 0.27657175 -0.89481485 -0.11592292 0.27030286 0.2780238 0.8047059 -0.34860256 -0.23718461 0.49289742 0.2757745 -0.2719815 -0.83650357 0.4279081 -0.34813493 -0.7843542 -0.29169905 0.78665775 0.8512063 0.5524962 0.6965423 -0.83091235 0.35204357 -0.29332906 -0.5411266 0.4143645 0.29879394 0.40416068 -0.43783253 -0.73355657 0.7020278 0.7078002 -1.0907903 -0.82440066 0.7423702 0.8138362 -0.42130232 0.7527003 0.5850441 -0.22044058 -0.7236471 -0.6447189 0.8670002 -0.70615524 -0.8628886 -0.40829352 -0.70032966 -0.770662 0.5221451 0.30932972 0.4952266 0.34399801 -0.46340755 0.68677074 0.6562086 -0.59302944 -0.4185523 -0.28875497 -0.36711028 0.5929899 -0.34999958 -0.81756115 0.84366345 -0.12790038; -0.41328105 0.8350619 -0.36371592 -0.5044399 -0.7386124 0.50419587 -0.033275973 -0.36038932 0.5268454 0.55895865 -0.16948646 -0.40610236 0.5874022 -0.84398997 -0.4155769 0.35589692 0.32351992 0.58732057 -0.61142606 0.9196675 0.681955 -0.9021308 0.29384643 -0.44132632 -0.6204438 0.7395101 0.62146306 -0.7394917 0.83430284 0.68471104 -0.6690516 0.65065324 -0.9957509 0.57958496 0.41631308 0.40327692 0.7773873 -0.882404 0.60267955 -0.29697087 -0.4117183 0.33700573 0.3454171 0.6898395 -0.54191196 -0.5327387 0.8649397 0.6309694 -0.5781122 0.12472181 -0.28869867 0.5801152 -0.41124424 0.8524347 0.4107315 -0.35287046 -0.87983686 -0.35534424 0.27192953 -0.98893154 -0.62490165 -1.0555962 -0.6559778 -0.48231396 -1.2060385 0.38443068 0.9874096 0.53793013 -0.55542344 0.89586955 0.4309594 -0.84370375 -0.6068061 -0.48002067 -0.4352499 0.9787741 -0.014543919 -0.93205935 0.4309669 -1.0362718], bias = Float32[-0.5232259; -0.59198105]))

loss_neuralode(p_trained_high)